package embl.ebi.trace;

import java.io.*;

/**
A superclass to define all the methods common to both of
the currently known trace formats. Parsing is done in the
subclasses and all display code is left to a viewer class.
@see ABIChromatogram
@see SCFChromatogram
*/

public abstract class Chromatogram {
	
	// Some handy constants
	public static final int	MAX_ASCII = 127;
	private static char[]	complementChars = new char[MAX_ASCII + 1];

	// Real chromatogram data
	int[]		A, C, G, T, basePosition;
	byte[]		quality;
	char		base[];
	String		comments;
	String		databaseName;
	String		accession;
	boolean		originalOrientation = true;

	public Chromatogram () {
		super();
		}

	public abstract void read(InputStream IN) throws IOException;
	public abstract void write(OutputStream OUT);
	public abstract int getTraceLength();
	public abstract int getBaseNumber();
	public abstract String getComments();


	public void setAccession(String sequenceName) {
		this.accession = sequenceName;
		}

	public String getAccession() {
		return (accession);
		}

	public void setDatabaseName(String dbName) {
		this.databaseName = dbName;
		}

	public String getDatabaseName() {
		return (databaseName);
		}

	public String getSequence() {
		return (new String(base));
		}

	public char getBase(int x) {
		return base[x];
		}

	public int getBasePosition(int x) {
		return basePosition[x];
		}

	public int getATrace(int x) {
		return A[x];
		}
	
	public int getCTrace(int x) {
		return C[x];
		}
	
	public int getGTrace(int x) {
		return G[x];
		}
	
	public int getTTrace(int x) {
		return T[x];
		}
	
	public int[] getATrace() {
		return A;
		}
		
	public int[] getCTrace() {
		return C;
		}
	
	public int[] getGTrace() {
		return G;
		}
	
	public int[] getTTrace() {
		return T;
		}

	public byte[] getQuality() {
		return quality;
		}

	public boolean getOriginalOrientation(){
		return (originalOrientation);
		}

	public void complement() {
		// Convert the sequence into its reverse and complement (opposite strand)
		// Must do this for the base calls, and the traces
		int[]	temp;

		// Easiest to switch is the quality values
		reverse_byteArray(quality);

		// Reverse the traces and complement
		reverse_intArray(A);
		reverse_intArray(C);
		reverse_intArray(G);
		reverse_intArray(T);
		temp = A;
		A = T;
		T = temp;
		temp = C;
		C = G;
		G = temp;
		temp = null;

		// Reverse base call position information
		reverse_intArray(basePosition);
		int lastSample =  getTraceLength() - 1;
		for (int i = 0; i < basePosition.length; i++) {
			basePosition[i] = lastSample - basePosition[i];
			}

		// Now the base characters themselves
		reverse_charArray(base);
		for (int i = 0; i < base.length; i++) {
			base[i] =  complement(base[i]);
			}

		originalOrientation = ! originalOrientation;

		return;
		}
	
	public static final void reverse_byteArray (byte[] arr){
		byte	temp;
		int	arrLast = arr.length - 1;
		int	midway = (arr.length) / 2;
		
		for (int i = 0; i < midway; i++){
			temp = arr[i];
			arr[i] = arr[arrLast - i];
			arr[arrLast - i] = temp;
			}
		return;
		}

	public static final void reverse_charArray (char[] arr){
		char	temp;
		int	arrLast = arr.length - 1;
		int	midway = (arr.length) / 2;
		
		for (int i = 0; i < midway; i++){
			temp = arr[i];
			arr[i] = arr[arrLast - i];
			arr[arrLast - i] = temp;
			}
		return;
		}

	public static final void reverse_intArray (int[] arr){
		int	temp;
		int	arrLast = arr.length - 1;
		int	midway = (arr.length) / 2;
		
		for (int i = 0; i < midway; i++){
			temp = arr[i];
			arr[i] = arr[arrLast - i];
			arr[arrLast - i] = temp;
			}
		return;
		}

	public static final char complement (char letter) {
		return (complementChars[(int)letter]);
		}
		
	static {
		// Initialise the complement chars table
		for (int i = 0; i <= Chromatogram.MAX_ASCII; i++){
			complementChars[i] = 'N';
			}
		complementChars['a'] = 't';
		complementChars['A'] = 'T';
		complementChars['t'] = 'a';
		complementChars['T'] = 'A';
		complementChars['u'] = 'a';
		complementChars['U'] = 'A';
		complementChars['c'] = 'g';
		complementChars['C'] = 'G';
		complementChars['g'] = 'c';
		complementChars['G'] = 'C';
		}

	/**
	Depending on chromatogram type (ABI or SCF), need to instantiate
	a different sub class of chromatogram. This method acts as a 
	chromatogram factory.
	*/
	public static Chromatogram createChromatogramSubType(InputStream streamIn)
		throws IOException {

		Chromatogram	chromData = null;

		DataInputStream magicDIN = new DataInputStream(streamIn);
		int magicNum = magicDIN.readInt();
		magicDIN.reset();
		switch(magicNum){
			case ABIChromatogram.MagicNum:
				chromData = new ABIChromatogram(streamIn);
				break;
			case SCFChromatogram.MagicNum:
				chromData = new SCFChromatogram(streamIn);
				break;
			default:
				throw new IOException("Unknown file type");
			}

		return(chromData);
		}
}

