/*
 * BLMain.java
 *
 * Created on November 7, 2007, 1:04 PM
 *
 * This is the file which contains all of the main classes, and functions for
 * running BioLegato.
 *
 * Current serializable number:
 * private static final long serialVersionUID = 7526472295622777040L;
 *
 * Released numbers:
 */
package org.biolegato.core.main;

import java.awt.Component;
import java.awt.Font;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.text.BadLocationException;
import org.biolegato.core.menu.GDEMakeMenus;
import org.biolegato.core.menu.GDEMenu;
import org.biolegato.core.plugins.PluginLoader;
import org.biolegato.core.pcdmenu.PCDMenu;
import org.biolegato.core.plugintypes.DataCanvas;
import org.biolegato.core.plugins.PluginWrapper;
import org.biolegato.core.properties.BLProperties;
import org.biolegato.core.properties.PropertiesListener;

/**
 * The main program and generic function class.
 * <p>
 *  This class is used to generate the main window, do all startup processing,
 *  and run the program.  This class also contains most of the utility functions.
 * </p>
 *
 * @version 0.7.3 6-Dec-2010
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public final class BLMain {

    public static enum OS {
	BSD,
	LINUX,
	HP_UX,
	MACOS,
	OSX,
	SOLARIS,
        WINDOWS_9X,
        WINDOWS_NT,
        UNIX;

        public boolean isWindows () {
            return (this == WINDOWS_9X || this == WINDOWS_NT);
        }

        public boolean isUNIX () {
            return (this != WINDOWS_9X && this != WINDOWS_NT);
        }
	
	/**
	 * Check out: http://lopica.sourceforge.net/os.html
	 */
        public static OS detectOS () {
            OS result = UNIX;
            String osName = System.getProperty("os.name").toLowerCase();

            if (osName.startsWith("windows")) {
                result = WINDOWS_NT;
                if (osName.startsWith("windows 9") || osName.equals("windows me")) {
                    result = WINDOWS_9X;
                }
            } else if (osName.startsWith("solaris") || osName.startsWith("sunos")) {
		result = SOLARIS;
            } else if (osName.startsWith("linux")) {
		result = LINUX;
            } else if (osName.endsWith("bsd")) {
		result = BSD;
            } else if (osName.startsWith("hp-ux")) {
		result = HP_UX;
            } else if (osName.startsWith("mac os")) {
		result = MACOS;
            } else if (osName.startsWith("mac os x")) {
		result = OSX;
	    } else {
		BLMain.warning("Did not detect your OS!  Defaulting to UNIX: " + System.getProperty("os.name")
		    + "   version: " + System.getProperty("os.version"));
	    }
            return result;
        }
    }

    public static enum ARCH {
	X86,
	AMD64,
	ALPHA,
	ARM,
	MIPS,
	SPARC,
	PPC,
	PPC64,
	UNKNOWN;

	/**
	 * Check out: http://lopica.sourceforge.net/os.html
	 */
        public static ARCH detectARCH () {
            ARCH result = UNKNOWN;
	    String osArch = System.getProperty("os.arch").toLowerCase();

            if (osArch.equals("sparc")) {
		result = SPARC;
            } else if (osArch.equals("arm")) {
		result = ARM;
            } else if (osArch.equals("alpha")) {
		result = ALPHA;
            } else if (osArch.equals("mips")) {
		result = MIPS;
            } else if (osArch.equals("amd64") || osArch.equals("x86_64")) {
		result = AMD64;
            } else if (osArch.equals("x86") || (osArch.startsWith("i") && osArch.endsWith("86"))) {
		result = X86;
            } else if (osArch.equals("ppc") || osArch.startsWith("power")) {
		if (!osArch.endsWith("64")) {
		    result = PPC;
		} else{
		    result = PPC64;
		}
	    } else {
		BLMain.warning("Did not detect your system archetecture: " + System.getProperty("os.arch"));
	    }
            return result;
        }
    }

/////////////////////////
//*********************//
//* PROGRAM CONSTANTS *//
//*********************//
/////////////////////////
    /**
     * This constant stores the program's name
     */
    public static final String NAME = "BioLegato";
    /**
     * This constant is set to the path of BioLegato.jar
     * The value of this constant determined at runtime.
     */
    public static final String PROGRAM_DIR =
            new File(
            BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()).isDirectory()
            ? BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()
            : new File(
            new File(
            BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()).getParent()).getPath();
    /**
     * This constant is set to the path of where biolegato was run
     * The value of this constant determined at runtime.
     */
    public static final String CURRENT_DIR = System.getProperty("user.dir");
    /**
     * This constant is used to keep track of version changes.
     */
    public static final String VERSION = "0.7.3";
    /**
     * The default font for BioLegato
     */
    public static final Font DEFAULT_FONT = new Font("Monospaced", Font.PLAIN, 12);
    /**
     * Stores the user's home directory
     */
    public static final String HOME_DIR = System.getProperty("user.home");
    /**
     * The amount of time between clicks to be considered a double click.
     */
    public static final int DOUBLE_CLICK_TIME = 300;
    /**
     * Stores the current operating system
     */
    public static final OS CURRENT_OS = OS.detectOS();
    /**
     * Stores the current machine architecture
     */
    public static final ARCH CURRENT_ARCH = ARCH.detectARCH();
    /**
     * This constant is used for Serialization
     */
    public static final long serialVersionUID = 7526472295622776147L;
    /**
     * The menu item for BioLegato's "About..."
     */
    public static final JMenuItem ABOUT_MENUITEM = new JMenuItem(new AbstractAction("About...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_A));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed(java.awt.event.ActionEvent evt) {
                aboutPopup();
            }
        });
    /**
     * The menu item for BioLegato's "Exit..."
     */
    public static final JMenuItem EXIT_MENUITEM = new JMenuItem(new AbstractAction("Exit") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_X));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed(java.awt.event.ActionEvent evt) {
                window.dispose();
            }
        });
/////////////////
//*************//
//* VARIABLES *//
//*************//
/////////////////
    /**
     * Reference to the main program window
     */
    private static JFrame window = null;
    /**
     * Stores the properties for BioLegato.  See BLProperties for a list of properties intrinsic to BioLegato.
     */
    private static BLProperties properties = null;
    /**
     * The tab pane storing all of the canvases.
     * This is stored so as to be able to get the current canvas for retrieving
     * the current data selection.
     */
    private static JTabbedPane canvasPane = null;
    /**
     * Stores the main menu for the program
     */
    private static JMenuBar menu = null;
    /**
     * The current directory to use for open and save file choosers.
     * This variable will cache the last opened directory, so as to maintain consistency.
     */
    private static File currentPWD = new File(CURRENT_DIR);
    /**
     * Stores all of the menu headings in BioLegato's main window.
     * This hashtable is used to add menu items to BioLegato's menu headings.
     * The key of the hashtable corresponds to the label of the menu heading.
     * The value of the hashtable is the menu heading's object.
     */
    private static Hashtable<String, JMenu> menuHeadings =
            new Hashtable<String, JMenu>();
    /**
     * Stores all of the plugins loaded into BioLegato
     */
    private static PluginLoader plugins = null;
    /**
     * Stores information regarding the usage of the debug command
     * NOTE: This should always be accessed using BLMain.debug
     */
    public static boolean debug = false;

/////////////////////////
//*********************//
//* STARTUP FUNCTIONS *//
//*********************//
/////////////////////////
    /**
     * Starts up BioLegato.
     *
     * @param args the command line arguments for BioLegato.
     */
    public static void main(String[] args) {
//////////////////////////
//**********************//
//* FUNCTION VARIABLES *//
//**********************//
//////////////////////////
        // RESERVED (general purpose - inner scoped)
        // fileIn
        // currentCanvas
        String value = "";	    // the value of the current argument parameter
        String argument = null;	    //
        List<File> dataAdd = new LinkedList<File>();

/////////////////////////
//*********************//
//* PROCESS ARGUMENTS *//
//*********************//
/////////////////////////

        // load the properties, plugins and file formats
        properties = new BLProperties();
        plugins = new PluginLoader(envreplace(getProperty("plugins")));

        // ensure that args is not null
        if (args != null) {
            // itterate through the command arguments
            for (String rawarg : args) {
                // discard null arguments
                if (rawarg != null) {
                    File fileIn = null;	    //
                    
                    // create a new file object to test if the argument is a file
                    fileIn = new File(rawarg);

                    // if the argument is a file name then read the file; otherwise,
                    // parse the argument
                    if (fileIn != null && fileIn.exists() && fileIn.isFile() &&
                            fileIn.canRead()) {
                        dataAdd.add(fileIn);
                    } else if (rawarg.startsWith("/") || rawarg.startsWith(
                            "-")) {
                        // copy the command line argument
                        argument = rawarg;

                        // trim the argument
                        if (argument.startsWith("--") && argument.length() > 2) {
                            argument = argument.substring(2);
                        } else if ((argument.startsWith("/") ||
                                argument.startsWith("-")) &&
                                argument.length() > 1) {
                            argument = argument.substring(1);
                        }

                        // If there is an equals '=' sign then the argument has a value.  Set the value
                        // variable to the arguments' value for further parsing
                        if (argument.indexOf('=') > 0) {
                            value = argument.substring(argument.indexOf('=') + 1);
                            argument = argument.substring(0, argument.indexOf(
                                    '='));
                        }

                        // make the argument lower case for better parsing
                        argument = argument.toLowerCase().trim();

                        // process the argument
                        if ("help".equals(argument) || "h".equals(argument) ||
                                "?".equals(argument)) {
                            ////////////
                            //********//
                            //* HELP *//
                            //********//
                            ////////////
                            // show BioLegato's usage
                            System.out.println(
                                    "Usage: biolegato [options] [files]\n" +
                                    "Use --optionlist  to see a detailed list of options\n" +
                                    "Use --manpage     to see BioLegato's manpage");
                            System.exit(0);
                        } else if ("optionlist".equals(argument)) {
                            ///////////////////
                            //***************//
                            //* OPTION LIST *//
                            //***************//
                            ///////////////////
                            // show BioLegato's list of options
                            System.out.println(
                                    "+-------------+------------------------------------------------------+\n" +
                                    "| Option      | Description                                          |\n" +
                                    "+-------------+------------------------------------------------------+\n" +
                                    "| help,?      | Displays usage information for BioLegato             |\n" +
                                    "| manpage,man | Displays the manpage entry for BioLegato             |\n" +
                                    "| optionlist  | Displays the list of options for using BioLegato     |\n" +
                                    "| version,v   | Displays the version information for BioLegato       |\n" +
                                    "| debug       | Enables BioLegato's debug mode                       |\n" +
                                    "| plugins     | Displays a list of loaded plugins                    |\n" +
                                    "| properties  | Displays a list of BioLegato's properties            |\n" +
                                    "+-------------+------------------------------------------------------+\n");
                            System.exit(0);
                        } else if ("manpage".equals(argument) ||
                                "man".equals(argument)) {
                            ///////////////
                            //***********//
                            //* MANPAGE *//
                            //***********//
                            ///////////////
                            // show BioLegato's usage
                            System.out.println(
                                    "NAME\n" +
                                    "    Bio Legato - A customizable GUI for running programs\n" +
                                    "\n" +
                                    "VERSION\n" +
                                    "    Version " + VERSION + "\n" +
                                    "\n" +
                                    "SYNOPSIS\n" +
                                    "    biolegato [options] [files]\n" +
                                    "\n" +
                                    "DESCRIPTION\n" +
                                    "    BioLegato is a customizable GUI for running programs.\n" +
                                    "    Its initial intent is to be a replacement for GDE; however\n" +
                                    "    with its large plugin API, it may be customized to run more\n" +
                                    "    than just CLI programs.\n" +
                                    "\n" +
                                    "    With the versitility of BioLegato's plugin interface, it\n" +
                                    "    supports a wide range of file formats which may be added to\n" +
                                    "    at any time through addition of plugins.\n" +
                                    "\n" +
                                    "OPTIONS\n" +
                                    "  NOTE: All command line parameters are case insensitive\n" +
                                    "        and may optionally begin with -, --, or / to prevent\n" +
                                    "        confusion with filenames\n" +
                                    "\n" +
                                    "    help,h,?\n" +
                                    "        Displays usage information for BioLegato\n" +
                                    "    manpage,man\n" +
                                    "        Displays the manpage entry for BioLegato (this screen)\n" +
                                    "    optionlist\n" +
                                    "        Displays the list of options for using BioLegato\n" +
                                    "    version,v\n" +
                                    "        Displays the version information for BioLegato\n" +
                                    "    debug\n" +
                                    "        Enables BioLegato's debug mode\n" +
                                    "    plugins\n" +
                                    "        Displays a list of loaded plugins\n" +
                                    "    properties\n" +
                                    "        Displays a list of BioLegato's properties\n" +
                                    "\n" +
                                    "USAGE EXAMPLES\n" +
                                    "    biolegato\n" +
                                    "\n" +
                                    "    biolegato insequence.gb\n" +
                                    "\n" +
                                    "    biolegato --debug\n" +
                                    "    biolegato /debug\n" +
                                    "    biolegato -debug\n" +
                                    "\n" +
                                    "    biolegato --plugins --properties --debug\n" +
                                    "\n" +
                                    "    biolegato --plugins /properties -debug insequence.gb\n" +
                                    "\n" +
                                    "ENVIRONMENT VARIABLES\n" +
                                    "    BioLegato searches for the following environment variables.  If they don't exist BioLegato sets them to defaults.\n" +
                                    "\n" +
                                    "      BL_HOME\n" +
                                    "        Default: the path of the current working directory\n" +
                                    "        Description: the path to reference BioLegato from\n" +
                                    "\n" +
                                    "      BL_MASKS\n" +
                                    "        Default: uses .blproperties value\n" +
                                    "        Description: the directory to populate the default colourmask\n" +
                                    "                     list\n" +
/*                                    "\n" +
				    "      BL_MENUS\n" +
				    "        Default: $BL_HOME/menus\n" +
				    "        Description: the locations to load the BioLegato menus from\n" +*/
                                    "\n" +
				    "      BL_PROPERTIES\n" +
				    "        Default: $BL_HOME/properties\n" +
				    "        Description: the locations to load the BioLegato properties file from\n" +
				    "\n" +
                                    "      GDE_HELP_DIR\n" +
                                    "        Default: null\n" +
                                    "        Description: specifies a path to search for the main .GDEmenus file\n" +
                                    "                     (NOTE: this is GDE_Menus.jar plugin specific)\n" +
                                    "\n" +
				    "      GDE_MAKEMENUS_DIR\n" +
				    "        Default: null\n" +
				    "        Description: specifies a path to use as the root for parsing GDE\n" +
				    "                     format menus in the same way as BIRCH's makemenus.py\n" +
                                    "\n" +
                                    "      BL_DEBUG\n" +
                                    "        Default: uses .blproperties value OR command switch --debug\n" +
                                    "        Description: set to \"true\" to enable debug mode\n" +
				    "\n" +
                                    "      NOTE: in properties files, BL_DIR points to the directory where\n" +
                                    "            BioLegato is stored\n" +
                                    "\n" +
                                    "PROPERTIES\n" +
                                    "    BioLegato supports the followind properties:\n" +
                                    "\n" +
                                    "      Property             Description\n" +
                                    "        debug                determines whether or not to display debug information\n" +
                                    "        plugins              sets the plugin directory\n" +
                                    "        temp                 sets the temporary files directory\n" +
                                    "        font.size            determines the default font size for objects in BioLegato\n" +
                                    "        font.type            determines the default font type for objects in BioLegato\n" +
                                    "        shell.name           determines the shell to execute\n" +
                                    "        shell.parameter      determines the parameter for directing the shell to execute a command\n" +
                                    "        pwd.properties       determines whether or not to read the properties file located in the current working directory\n" +
                                    "        user.properties      determines whether or not to read the user's propery files (located in the user's home directory\n" +
                                    "        user.plugins         determines whether or not to read the user's plugin files (located in the user's home directory\n" +
                                    "        default.canvas       determines the default canvas to show on startup\n" +
                                    "        default.fileformat   determines the default file format for open/save dialogs\n" +
                                    "        GDE.menu             determines whether or not to read the GDE menu files (backwards compatibility mode)\n" +
                                    "        GDE.help.viewer      determines which file to feed the help file to\n" +
                                    "        GDE.help.path        the location to search for the help files\n" +
                                    "        undo                 whether or not to enable undo support\n" +
                                    "        undo.size            whether or not to enable undo support\n" +
                                    "\n" +
                                    "    This properties class will read properties files from the directory containing BioLegato, the user directory and finally\n" +
                                    "    the directory BioLegato was launched from.  Please note that you can disable reading of properties files\n" +
                                    "    from the user directory via. \"user.properties\"; likewise you can disable reading properties files in the directory BioLegato\n" +
                                    "    was launched from via. \"pwd.properties\"" +
                                    "\n" +
                                    "    NOTE: for path properties BioLegato will replace all $'s with the appropriate environment variables if set.\n" +
                                    "\n" +
                                    "SUPPORTED PLUGINS\n" +
                                    "    Currently BioLegato supports the following main types of Plugins:\n" +
                                    "\n" +
                                    "        Canvases\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.DataCanvas\n" +
                                    "        File formats\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.DataFormat\n" +
                                    "        Menu types\n" +
                                    "           extends the class: org.biolegato.core.plugintypes.MenuType\n" +
                                    "\n" +
                                    "    For more information about plugins, please consult the BioLegato API\n" +
                                    "\n" +
                                    "FILES\n" +
                                    "    biolegato\n" +
                                    "        Script to run BioLegato\n" +
                                    "    biolegato.jar\n" +
                                    "        BioLegato's core java code\n" +
                                    "    changelog.txt\n" +
                                    "        BioLegato's revision history\n" +
                                    "    plugins/GDE.jar" +
                                    "        Provides a GDE style multiple alignment canvas and related support\n" +
                                    "    plugins/tables.jar" +
                                    "        Provides tabulated data support\n" +
                                    "\n" +
                                    "FILE FORMATS\n" +
                                    "    All file formats in BioLegato are supported through plugins.\n" +
                                    "    Below is a list of file formats supported by a default\n" +
                                    "    installation of BioLegato (with all standard plugins):\n" +
                                    "\n" +
                                    "        BioLegato flatfiles\n" +
                                    "        FastA files\n" +
                                    "        GDE flatfiles\n" +
                                    "        GDE format files\n" +
                                    "        GenBank files (2008 standard compliant)\n" +
                                    "\n" +
                                    "BUGS\n" +
                                    "    There are currently no known bugs in this version of BioLegato\n" +
                                    "\n" +
                                    "    Please report all bugs to: Graham Alvare <alvare@cc.umanitoba.ca>" +
                                    "\n" +
                                    "AUTHORS\n" +
                                    "    Dr. Brian Fristensky\n" +
                                    "    Department of Plant Science\n" +
                                    "    University of Manitoba\n" +
                                    "    Winnipeg, MB  Canada R3T 2N2\n" +
                                    "\n" +
                                    "    Email: frist@cc.umanitoba.ca\n" +
                                    "    Web: http://home.cc.umanitoba.ca/~frist\n" +
                                    "\n" +
                                    "    Graham Alvare\n" +
                                    "    Department of Plant Science\n" +
                                    "    University of Manitoba\n" +
                                    "    Winnipeg, MB  Canada R3T 2N2\n" +
                                    "\n" +
                                    "    Email: alvare@cc.umanitoba.ca\n" +
                                    "    Web: http://home.cc.umanitoba.ca/~alvare\n");
                            System.exit(0);
                        } else if ("version".equals(argument) ||
                                "v".equals(argument)) {
                            ///////////////
                            //***********//
                            //* VERSION *//
                            //***********//
                            ///////////////
                            // show BioLegato's version
                            System.out.println("BioLegato v" + VERSION);
                            System.exit(0);
                        } else if ("debug".equals(argument)) {
                            /////////////
                            //*********//
                            //* DEBUG *//
                            //*********//
                            /////////////
                            // enable debug information
                            debug = true;
                        } else if ("plugins".equals(argument)) {
                            ///////////////
                            //***********//
                            //* PLUGINS *//
                            //***********//
                            ///////////////
                            // show BioLegato's loaded plugins
                            System.out.println("(Current plugins path: " +
                                    getProperty("plugins") + ")");
                            System.out.println("-- listing plugins loaded --");
                            String[] pluginList =
                                    plugins.getPluginHash().keySet().toArray(
                                    new String[0]);
                            for (String pluginName : pluginList) {
                                System.out.println("Plugin: " + pluginName);
                            }
                            System.out.println("-- end of plugin list --");
                        } else if ("properties".equals(argument)) {
                            System.out.println("**********");
                            System.out.println("PROPERTIES");
                            System.out.println("**********");
                            //////////////////
                            //**************//
                            //* PROPERTIES *//
                            //**************//
                            //////////////////
                            // show BioLegato's current settings
                            properties.list(System.out);
                        } else {
                            BLMain.error("Unknown argument: " + rawarg,
                                    "processArgs");
                        }
                    }
                }
            }
        }
        // if debug is set, display the command
        if (BLMain.debug) {
            BLMain.message("Command line arguments read successfully",
                    "processArgs");
        }
        
	if ("true".equalsIgnoreCase(BLMain.getProperty("debug")) || (System.getenv("BL_DEBUG") != null
		&& !"".equals(System.getenv("BL_DEBUG").trim())
		&& !"false".equalsIgnoreCase(System.getenv("debug")))) {
	    debug = true;
	}

////////////////////////
//********************//
//* CONFIGURE WINDOW *//
//********************//
////////////////////////
        canvasPane = new JTabbedPane();
        window = new JFrame(NAME);

        // add tabbed pane to window
        menu = new JMenuBar();
        Box displayBox = new Box(BoxLayout.PAGE_AXIS);
        window.setJMenuBar(menu);
        window.add(canvasPane);

////////////
//********//
//* MENU *//
//********//
////////////
        
        addMenuHeading("File");
        
        ////////////////////////////////
        //****************************//
        //* READ IN THE CUSTOM MENUS *//
        //****************************//
        ////////////////////////////////

        /*************************************************************
         * ittearates through the Plugins searching for menu formats *
         *************************************************************/
        PCDMenu.loadMenu();
        GDEMenu.loadMenu();
        GDEMakeMenus.loadMenu();

        ///////////////////////////////////////////
        //***************************************//
        //* ADD THE DEFAULT TRAILING MENU ITEMS *//
        //***************************************//
        ///////////////////////////////////////////

        /**************************
         * Add the "About" button *
         **************************/
        addMenuItem("Help", ABOUT_MENUITEM);	/* Event handler - open about it window */

        /*************************
         * Add the "Exit" button *
         *************************/
        addMenuItem("File", EXIT_MENUITEM);	/* Event handler - exit the program */

////////////////
//************//
//* CANVASES *//
//************//
////////////////
        // Load the canvas panes.
        // Populates the tabbed pane with all plugin canvases
        for (PluginWrapper plugin : getPlugins(DataCanvas.class)) {
            try {
                if (plugin.isA(Component.class)) {
                    DataCanvas currentCanvas = null;    // used for loading canvases (for testing proper object creation)
                    
                    // adds the valid canvas
                    currentCanvas = (DataCanvas) plugin.create();
                    if (currentCanvas != null) {
                        canvasPane.add(currentCanvas.getTabName(), (Component) currentCanvas);
                        if (currentCanvas.getTabName() != null &&
                                currentCanvas.getTabName().equalsIgnoreCase(getProperty(
                                "default.canvas"))) {
                            canvasPane.setSelectedComponent((Component) currentCanvas);
                        }
                    }
                    
                    for (File file : dataAdd) {
                        currentCanvas.readFile("", file);
                    }
                }
            } catch (Throwable th) {
                BLMain.error("error loading the plugin: " + plugin.getName(), "BLMain");
                th.printStackTrace();
            }
        }

///////////////
//***********//
//* DISPLAY *//
//***********//
///////////////

        // center and draw the frame on the screen
        window.pack();
        window.setVisible(true);
        window.setLocationRelativeTo(null);
        window.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    }

////////////////////////////
//************************//
//* PROPERTIES FUNCTIONS *//
//************************//
////////////////////////////
    /**
     * Changes properties for BioLegato.
     *
     * @param key the property to change.
     * @param value the new value for the property.
     */
    public static void setProperty(String key, String value) {
        properties.setProperty(key, value);
    }

    /**
     * Retrieves individual settings for BioLegato.
     * This is used to obtain values of properties in BioLeagato
     *
     * @param property the property key to retrieve the value for
     * @return the property value corresponding to the key parameter
     */
    public static String getProperty(String property) {
        return properties.getProperty(property);
    }

//////////////////////
//******************//
//* DATA FUNCTIONS *//
//******************//
//////////////////////
    /**
     * Returns the current/selected data in the canvas.
     *
     * @return the current data for usage by commands
     * @throws BadLocationException any exceiptions related to obtaining the data
     */
    public static DataCanvas getCanvas() throws BadLocationException {
        return (DataCanvas) (canvasPane.getSelectedComponent());
    }

////////////////////////
//********************//
//* WINDOW FUNCTIONS *//
//********************//
////////////////////////
    /**
     * Provides access for other classes to the main program window JFrame object.
     **
     * @return the JFrame object corresponding to the main program window.
     */
    public static JFrame getJFrame() {
        return window;
    }
    
    /**
     * Returns the properties object associated with this instance of BioLegato
     **
     * @return the properties object associated
     */
    public static BLProperties getProperties() {
        return properties;
    }

    /**
     * Adds a menu heading (JMenu) to our menu (BLMenu).
     **
     * @param name the name of the menu heading
     * @return either the JMenu which was added or the JMenu that corresponds to the existing menu heading.
     */
    public static JMenu addMenuHeading(String name) {
        JMenu heading = menuHeadings.get(name);	// the JMenu object to add

        // check if the heading already exists in the menu. (null indicates that the menu heading is new)
        if (heading == null) {
            // create a new menu heading object
            heading = new JMenu(name);

            // set the mnemonic
            if (name != null && name.length() >= 1 &&
                    ((name.charAt(0) >= 'a' && name.charAt(0) <= 'z') ||
                    (name.charAt(0) >= 'A' && name.charAt(0) <= 'Z'))) {
                heading.setMnemonic(name.charAt(0));
            }

            // add the heading
            menuHeadings.put(name, heading);
            menu.add(heading);
        }

        // return the menu heading object
        return heading;
    }

    /**
     * Adds a menu heading (JMenu) to our menu (BLMenu).
     **
     * @param order the position to place the menu tag
     * @param name the name of the menu heading
     * @return either the JMenu which was added or the JMenu that corresponds to the existing menu heading.
     */
    public static JMenu addMenuHeading(int order, String name) {
        JMenu heading = addMenuHeading(name);	// the hading to add the item to.

        // ensure that the menu heading is in the correct order
        if (menu.getComponentIndex(heading) != order) {
            menu.remove(heading);
            menu.add(heading, order);
        }
        return heading;
    }

    /**
     * Adds a menu item (JMenuItem) to a menu heading (JMenu) within the window (ProgramWindow)
     * <p>If the heading which was entered does not exist, this function will create it;
     *	hence why addMenuHeading returns the heading corresponding to the name entered if a
     *	menu heading with that name already exists.</p>
     **
     * @param location is the location in the menu to insert the item at
     * @param headingName is the name of the menu heading to add the menu item to.
     * @param menuItem is the menu item to add to the heading.
     */
    public static void addMenuItem(int location, String headingName, JMenuItem menuItem) {
        // enforce that the menu heading exists and then add the menu item to it
        addMenuHeading(headingName).insert(menuItem, location);
    }
    /**
     * Adds a menu item (JMenuItem) to a menu heading (JMenu) within the window (ProgramWindow)
     * <p>If the heading which was entered does not exist, this function will create it;
     *	hence why addMenuHeading returns the heading corresponding to the name entered if a
     *	menu heading with that name already exists.</p>
     *
     * @param headingName is the name of the menu heading to add the menu item to.
     * @param menuItem is the menu item to add to the heading.
     */
    public static void addMenuItem(String headingName, JMenuItem menuItem) {
        // enforce that the menu heading exists and then add the menu item to it
        addMenuHeading(headingName).add(menuItem);
    }

    /**
     * Displays an about popup for BioLegato.
     */
    public static void aboutPopup() {
        JOptionPane.showMessageDialog(window,
                "BioLegato version " +
                VERSION +
                "\nby Graham Alvare and Brian Fristensky\nUniveristy of Manitoba 2008-2010",
                "About BioLegato",
                JOptionPane.QUESTION_MESSAGE);
    }

    /**
     * Adds a properties listener to the properties object.
     *
     * @param key the key of the properties to listen to.
     * @param listener the listener to add to the object.
     */
    public static void addPropertiesListener(String key, PropertiesListener listener) {
        properties.addPropertiesListener(key, listener);
    }

/////////////////////////
//*********************//
//* GENERAL FUNCTIONS *//
//*********************//
/////////////////////////
    /**
     * Ensures that the command will be executed properly as a shell command
     * <p>This function generates a command list for execution.  The command list will contain
     *	the appropriate shell for the current operating system, followed by the "execution-argument",
     *	(whatever flag is required to tell the shell that the rest of the commandline should be executed
     *	by the shell), followed by the command to execute (the variable cmd)</p>
     * <p>Operating system values obtained from http://lopica.sourceforge.net/os.html</p>
     *
     * @param cmd the command string to execute
     * @return if successful a Process object for the executed command.
     */
    public static Process safeExecute(String cmd) {
        Process result = null;                          // the result of the function call
        StringBuffer message = new StringBuffer();	// used for printing debug information
        String[] execute = new String[]{cmd};          // default case - run the command by itself

        // builds the execution array
        if (properties.getProperty("shell.name") != null) {
            // if there is a shell for the current operating system, execute the command through the shell
            if (properties.getProperty("shell.parameter") != null) {
                execute =
                        new String[]{properties.getProperty("shell.name"),
                            properties.getProperty("shell.parameter"), cmd};
            } else {
                execute = new String[]{properties.getProperty("shell.name"),
                            cmd};
            }
        }

        // if debug is set, display the command
        if (BLMain.debug) {
            for (String parameter : execute) {
                // NOTE: append is faster than + or concat operators
                message.append(parameter + " ");
            }
            // relay the command string the message system
            BLMain.message(message.toString(), "safeExecute");
        }

        // obtain the process object for the command
        try {
            result = Runtime.getRuntime().exec(execute);
        } catch (Throwable e) {
            e.printStackTrace();
        }

        // return the resulting process object
        return result;
    }

    /**
     * Runs simple shell commands.
     * Reroutes all output to the console.
     *
     * @param cmd the command string to run
     * @param data the data to use as standard input (System.in)
     */
    public static void shellCommand(String cmd, String data) {
        int result = -65535;			    // the result of running the command
        final Process process = safeExecute(cmd);   // the process object obtained on execution

        // ensure the process object is not null
        if (process != null) {
            try {
                // passes any data to the program via standard in
                if (data != null) {
                    (new OutputStreamWriter(process.getOutputStream())).write(
                            data);
                }

                // this thread object prints the output from the program
                new Thread() {

                    @Override
                    public void run() {
                        String line = "";
                        BufferedReader reader = null;
                        try {
                            reader = new BufferedReader(new InputStreamReader(
                                    process.getInputStream()));
                            while ((line = reader.readLine()) != null) {
                                System.out.println(line);
                            }
                        } catch (IOException ioe) {
                            ioe.printStackTrace();
                        }
                    }
                }.start();
                // this thread object prints the error output from the program
                new Thread() {

                    @Override
                    public void run() {
                        String line = "";
                        BufferedReader reader = null;
                        try {
                            reader = new BufferedReader(new InputStreamReader(
                                    process.getErrorStream()));
                            while ((line = reader.readLine()) != null) {
                                System.err.println(line);
                            }
                        } catch (IOException ioe) {
                            ioe.printStackTrace();
                        }
                    }
                }.start();

                // display the command's result if debug is enabled
                if (BLMain.debug) {
                    BLMain.message("BioLegato: Command executed successfully, returned: " +
                            process.waitFor(), "shellCommand");
                } else {
                    process.waitFor();
                }
            } catch (Throwable e) {
                // if there are any errors, print them to the error prompt
                BLMain.error("BioLegato: error executing command: " + cmd,
                        "shellCommand");
                e.printStackTrace();
            }
        }
    }

    /**
     * Sends an error message to BioLegato's standard err.
     * Alias for error(message, null);
     *
     * @param message the error message to send.
     */
    public static void error(String message) {
        error(message, null);
    }

    /**
     * Sends an error message to BioLegato's standard err.
     *
     * @param message the error message to send.
     * @param location the location the error occurred.
     */
    public static void error(String message, String location) {
        // print the error to the error stream
        System.err.println(NAME + ((location != null && !"".equals(location.trim()))
                ? " (" + location + ")" : "") + ": ERROR --- " + message);
    }

    /**
     * Sends an warning message to BioLegato's standard err.
     * Alias for warning(message, null);
     *
     * @param message the warning message to send.
     */
    public static void warning(String message) {
        warning(message, null);
    }

    /**
     * Sends an warning message to BioLegato's standard err.
     *
     * @param message the warning message to send.
     * @param location the location the error occurred.
     */
    public static void warning(String message, String location) {
        // prints the warning to BioLegato's error stream
        System.err.println(NAME + ((location != null && !"".equals(location.trim()))
                ? " (" + location + ")" : "") + ": WARNING --- " + message);
    }

    /**
     * Sends a message to BioLegato's standard out.
     * Alias for message(message, null);
     *
     * @param message the message to send.
     */
    public static void message(String message) {
        message(message, null);
    }

    /**
     * Sends a message to BioLegato's standard out.
     *
     * @param message the message to send.
     * @param location the location the message was sent from.
     */
    public static void message(String message, String location) {
        // prints the warning to BioLegato's error stream
        System.out.println(NAME + ((location != null && !"".equals(location.trim()))
                ? " (" + location + ")" : "") + ": " + message);
    }

    /**
     * Reads the complete contents of a BufferedReader into a string.
     *
     * @param reader the BufferedReader to read.
     * @return the contents of the BufferedReader.
     * @throws IOException throws any exceptions from the read operation.
     */
    public static String readStream(BufferedReader reader) throws IOException {
        String line = reader.readLine();            // the current line to read
        StringBuffer result = new StringBuffer();   // the result of the operation

        // read in every line from the stream
        // NOTE: append is faster than + or concat operators
        while (line != null) {
            // don't forget to add the \n back to the line
            result.append(line).append("\n");

            // itterate to the next line
            line = reader.readLine();
        }

        // return the resulting string
        return result.toString();
    }

    /**
     * Checks if a character array is all digits.
     *
     * @param test the character array to test.
     * @return true if the array only contains digits.
     */
    public static boolean testNumber(char[] test) {
        boolean result = false;

        if (test != null && test.length > 0) {
            result = true;
            for (int count = 0; result && count < test.length; count++) {
                result &= Character.isDigit(test[count]);
            }
        }
        return result;
    }

    /**
     * Replaces $VARIABLE name with environment variables.
     * This function provides functionality similar to BASH.
     *
     * NOTE: if the variable is $BL_HOME, and this is not set, 
     *	the variable will default to the current working directory.
     *
     * @param change the string to modify.
     * @return the modified string.
     */
    public static String envreplace(String change) {
        int start = 0;
        int end = -1;
        String replace = null;
        String variable = "";

        if (change != null) {
            while ((start = change.indexOf('$', start)) >= start && start > -1) {
                for (end = start + 1; end < change.length() && (change.charAt(end) == '_' || Character.isLetter(change.charAt(end))); end++) {
                    /* ITTERATE */
                }

                // get the information to perform the string replacement.
                variable = change.substring(start + 1, end);
                replace = System.getenv(variable);

                // ensure BL_HOME is set properly.
                if (variable.equalsIgnoreCase("BL_DIR")) {
                    replace = BLMain.PROGRAM_DIR;
                }
                
                // ensure BL_HOME is set properly.
                if (variable.equalsIgnoreCase("BL_HOME")) {
                    if (replace == null || "".equals(replace.trim()) || !new File(replace).exists()) {
                        replace = BLMain.CURRENT_DIR;
                    } else {
                        replace = new File(replace).getAbsolutePath();
                    }
                }

                // perform the string replacement.
                if (replace != null) {
                    change = change.substring(0, start) + replace + change.substring(end);
                } else {
                    start++;
                }
            }
        }
        return change;
    }

    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     *
     * @param test the character to test.
     * @return true if the character is not a digit, otherwise false.
     */
    public final static boolean notDigit(char test) {
        return ((test < '0' || test > '9') && (test < 'a' || test > 'z') && (test < 'A' || test > 'Z'));
    }

    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     *
     * @param test the character to test.
     * @return true if the character is a digit, otherwise false.
     */
    public final static boolean isDigit(char test) {
        return ((test >= '0' && test <= '9') || (test >= 'a' && test <= 'z') || (test >= 'A' && test <= 'Z'));
    }
    
    /**
     * Gets the present working directory "PWD" for BioLegato (this does NOT change CURRENT_DIR!)
     * This value can be used to cache which directory file dialog boxes should use as their
     * "current working directory"
     *
     * @return the current value of "PWD"
     */
    public static File getCurrentPWD() {
	return currentPWD;
    }
    /**
     * Sets the present working directory "PWD" for BioLegato (this does NOT change CURRENT_DIR!)
     * This value can be used to cache which directory file dialog boxes should use as their
     * "current working directory"
     *
     * @param newPWD the new directory to use as the "PWD"
     */
    public static void setCurrentPWD(File newPWD) {
	currentPWD = newPWD;
    }
    /**
     * Returns a list with all of the plugins loaded by BioLegato that are
     * instances of the class type indicated.
     **
     * @param type the plugin type to find
     * @return a list containing all of the plugins loaded
     */
    public static Collection<PluginWrapper> getPlugins(Class<?> type) {
        return plugins.getPlugins(type);
    }

}
