package org.biolegato.core.pcdmenu.widgets;

import java.awt.Component;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.biolegato.core.pcdmenu.*;

/**
 * Wraps a widget(s) for selecting numeric values.
 **
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class NumberWidget implements Widget, ChangeListener {
    /**
     * The minimum number allowed
     */
    private int min;
    /**
     * The maximum number allowed
     */
    private int max;
    /**
     * The default for the number field
     */
    private int value;
    /**
     * The label for the widget
     */
    private String label = null;
    /**
     * The slider portion of the number selector widget.
     */
    private JSlider slider = null;
    /**
     * The slider portion of the number selector widget.
     */
    private JSpinner spinner = null;
    /**
     * Used for serialization purposes.
     */
    private static final long serialVersionUID = 7526472295622776161L;


    /**
     * Creates a new instance of EZSlider
     * 
     * @param data the map containing all of the information required to make the number selector.
     */
    public NumberWidget (String label, int min, int max, int value) {
        this.label = label;
        this.min = min;
        this.max = max;
        this.value = value;
    }

    /**
     * Handles synchronization between the slider and the spinner.
     * 
     * @param e used to determine which widget was changed.
     */
    public void stateChanged (ChangeEvent e) {
        if (e.getSource() instanceof JSpinner) {
            value = ((Number) spinner.getValue()).intValue();
        } else {
            value = slider.getValue();
        }
        slider.setValue(value);
        spinner.setValue(value);
    }

    public Component display() {
        Box container = new Box(BoxLayout.LINE_AXIS);

        if (slider == null && spinner == null) {
            // create the spinner
            spinner = new JSpinner(new SpinnerNumberModel(value, min, max, 1));
            spinner.addChangeListener(this);

            // create the slider
            slider = new JSlider(JSlider.HORIZONTAL, min, max, value);
            slider.addChangeListener(this);

            // add the spinner
            container.add(spinner);
            container.add(new JLabel("" + min));
        }

        // add the slider
        container.add(slider);
        container.add(new JLabel("" + max));

        return container;
    }

    public void close() {
        slider = null;
        spinner = null;
    }

    /**
     * Returns the current value of the variable.
     *
     * @return the value selected by the slider/spinner.
     */
    public Object getValue () {
        return value;
    }
}

