package org.biolegato.gdesupport.canvas;

/*
 * BLGDECanvas.java
 *
 * Created on December 3, 2007, 1:56 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.List;
import java.util.Stack;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.data.GDEModel;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.core.main.BLMain;
import org.biolegato.core.plugintypes.DataCanvas;
import org.biolegato.gdesupport.formats.DataFormat;
import org.biolegato.core.properties.PropertiesListener;
import org.biolegato.gdesupport.canvas.list.GDEList;
import org.biolegato.gdesupport.canvas.listeners.CursorListener;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import org.biolegato.gdesupport.canvas.textarea.BLTextArea;
import org.biolegato.gdesupport.canvas.textarea.GDETextArea;
import org.biolegato.gdesupport.canvas.textarea.UndoableGDETextArea;
import org.biolegato.gdesupport.canvas.undo.Undoable;
import org.biolegato.gdesupport.formats.FastAFile;
import org.biolegato.gdesupport.formats.GDEFile;
import org.biolegato.gdesupport.formats.GDEFlatfile;
import org.biolegato.gdesupport.formats.GenBankFile2008;

/**
 * The GDE-style sequence-based canvas class
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDECanvas extends Box implements DataCanvas, ListSelectionListener, CursorListener, ModeListener, PropertiesListener {

    /**
     * The current dataset for the GDE canvas
     */
    private final GDEModel datamodel = new GDEModel();
    /**
     * Used for selecting genes
     */
    private GDEList nameList = new GDEList(this);
    /**
     * The text area for data manipulation
     */
    private GDETextArea dataCollector = null;
    /**
     * The alternate text area for data manipulation (for splits)
     */
    private GDETextArea altDataCollector = null;
    /**
     * Stores which canvas contains the cursor and any current data selections.
     */
    private GDECanvasObject currentPane = nameList;
    /**
     * Used for serialization purposes
     */
    private static final long serialVersionUID = 7526472295622776154L;
    /**
     * The scroll pane containing the main canvas
     */
    private JScrollPane dataPane;
    /**
     * The scroll pane containing the split canvas
     */
    private JScrollPane altDataPane;
    /**
     * The box to store all split canvases
     */
    private Box canvasPane = new Box(BoxLayout.LINE_AXIS);
    /**
     * The split pane for split canvases
     */
    private JSplitPane splitPane;
    /**
     * Reference pointer to self
     */
    private final GDECanvas canvasSelf = this;
    /**
     * Stores the row and column position status
     */
    private final JLabel status = new JLabel("Row: 1 Col: 1");
    /**
     * Stores the insertion mode status
     */
    private final JLabel insertStatus = new JLabel("     ");
    /**
     * The stack of undoable events.
     * <p>
     *  Every time an event occurs within the BLTextArea that is undoable, an <b>undo object</b>
     *  is created and added to this stack.  This allows one to undo performed SeqDoc actions.
     *  To undo an action, pop the top object off of the stack and call its <b>undo method</b>.
     *  This will perform the undo and the <b>undo method</b> will return a <b>redo object</b>.
     *  The resulting <b>redo object</b> can then be added to the <b>redoStack</b> stack.
     * </p>
     */
    private Stack<Undoable> undoStack = new Stack<Undoable>();
    /**
     * The stack of redoable events
     * <p>
     *  Every time an event is undone, a redo object is created and added to this stack.
     *  A <b>redo object</b> is essentially an <b>undo object</b> created to "undo an undo".
     *  To perform a redo, pop the top object off the stact and call its <b>undo method</b>.
     *  This will perform the redo and the <b>undo method</b> will return an <b>undo object</b>.
     *  The resulting <b>undo object</b> can then be added to the <b>undoStack</b> stack.
     * </p>
     */
    private Stack<Undoable> redoStack = new Stack<Undoable>();
    /**
     * Action for splitting the canvas
     */
    private final AbstractAction splitAction = new AbstractAction("Split") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    
	    if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
		altDataCollector = new UndoableGDETextArea(canvasSelf);
	    } else {
		altDataCollector = new GDETextArea(canvasSelf);
	    }
	    
	    canvasPane.remove(dataPane);
	    
	    altDataPane = new JScrollPane(altDataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
	    splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, dataPane, altDataPane);
	    splitPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
	    splitPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);
	    canvasPane.add(splitPane);
	    
            dataCollector.removePopupMenuItem(splitMenuItem);
            dataCollector.addPopupMenuItem(joinMenuItem);
	    
            altDataCollector.addPopupMenuItem(new JMenuItem(joinAction));
        }

    };
    /**
     * Action for joining split canvases
     */
    public final AbstractAction joinAction = new AbstractAction("Join") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    if (currentPane == altDataCollector) {
		currentPane = dataCollector;
	    }
	    if (splitPane != null) {
		canvasPane.remove(splitPane);
		splitPane = null;
		altDataPane = null;
		canvasPane.add(dataPane);
	    }
	    
            dataCollector.removePopupMenuItem(joinMenuItem);
            dataCollector.addPopupMenuItem(splitMenuItem);
        }

    };
    /**
     * The action "Cut"
     */
    public final AbstractAction cutAction = new AbstractAction("Cut") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_C));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.cut();
        }

    };
    /**
     * The action "Copy"
     */
    public final AbstractAction copyAction = new AbstractAction("Copy") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_Y));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.copy();
        }

    };
    /**
     * The action "Paste"
     */
    public final AbstractAction pasteAction = new AbstractAction("Paste") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.paste();
        }

    };
    /**
     * The action "Undo"
     */
    public final AbstractAction undoAction = new AbstractAction("Undo") {

        private static final long serialVersionUID = 7526472295622777038L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_U));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.undo();
        }

    };
    /**
     * The action "Redo"
     */
    public final AbstractAction redoAction = new AbstractAction("Redo") {

        private static final long serialVersionUID = 7526472295622777039L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_R));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.redo();
        }

    };
    /**
     * The action "Change Case"
     */
    public final AbstractAction changeCaseAction = new AbstractAction("Change case") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/



        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_E));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            canvasSelf.changeCase();
        }
    };


    /**
     * The menu item "Select sequence by name"
     */
    private final JMenuItem selectByNameMenuItem = new JMenuItem(nameList.selectByNameAction);
    /**
     * Action for splitting the canvas
     */
    private final JMenuItem splitMenuItem = new JMenuItem(splitAction);
    /**
     * The menu item for joining split canvases
     */
    private final JMenuItem joinMenuItem = new JMenuItem(joinAction);
    /**
     * The menu itemaction "Cut"
     */
    private final JMenuItem cutMenuItem = new JMenuItem(cutAction);
    /**
     * The menu item "Copy"
     */
    private final JMenuItem copyMenuItem = new JMenuItem(copyAction);
    /**
     * The menu item "Paste"
     */
    private final JMenuItem pasteMenuItem = new JMenuItem(pasteAction);
    /**
     * The menu item "Undo"
     */
    private final JMenuItem undoMenuItem = new JMenuItem(undoAction);
    /**
     * The menu item "Redo"
     */
    private final JMenuItem redoMenuItem = new JMenuItem(redoAction);
    /**
     * The menu item "Change Case"
     */
    private final JMenuItem changeCaseMenuItem = new JMenuItem(changeCaseAction);
    /**
     * The menu item "Select group"
     */
    private final JMenuItem selectGroupMenuItem = new JMenuItem(nameList.selectGroupAction);
    /**
     * The menu item "Select All"
     */
    private final JMenuItem selectAllMenuItem = new JMenuItem(nameList.selectAllAction);
    /**
     * The menu item for grouping sequences in the canvas
     */
    private final JMenuItem groupMenuItem = new JMenuItem(nameList.groupAction);
    /**
     * The menu item for ungrouping sequences in the canvas
     */
    private final JMenuItem ungroupMenuItem = new JMenuItem(nameList.ungroupAction);
    /**
     * The menu item for "Get Info..."
     */
    private final JMenuItem getInfoMenuItem = new JMenuItem(nameList.getInfoAction);
    /**
     * Self-reference for inner classes.
     */
    public final GDECanvas gdeCanvasSelf = this;
    

    /**
     * Creates a new instance of BLGDECanvas
     */
    public GDECanvas () {
        super(BoxLayout.PAGE_AXIS);
        
        int menuInsertCount = 0;

        BLMain.setProperty("font.bold", "false");
        
        //////////////////////////////////////
        //**********************************//
        //* ADD THE DEFAULT TOP MENU ITEMS *//
        //**********************************//
        //////////////////////////////////////

        /*************************
         * Add the "Open" button *
         *************************/
        BLMain.addMenuItem(0, "File", new JMenuItem(new AbstractAction("Open...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_O));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed(java.awt.event.ActionEvent evt) {           /* Event handler - open the file */
                JFileChooser openDialog = new JFileChooser();
		File[] openFiles;

                openDialog.setCurrentDirectory(BLMain.getCurrentPWD());
                openDialog.setAcceptAllFileFilterUsed(true);
		openDialog.setMultiSelectionEnabled(true);
                org.biolegato.gdesupport.formats.DataFormat.addFormats(openDialog,
                        BLMain.getProperty("default.fileformat").toString());
                // if a file is selected, open it
                if (openDialog.showOpenDialog(BLMain.getJFrame()) ==
                        JFileChooser.APPROVE_OPTION) {
		    if (openDialog.getSelectedFiles() != null) {
			openFiles = openDialog.getSelectedFiles();
			for (File ofile : openFiles) {
			    if (ofile.exists() && ofile.isFile()) {
				try {
				    if (openDialog.getFileFilter() != null && !openDialog.getFileFilter().equals(
					    openDialog.getAcceptAllFileFilter())) {
                                        datamodel.addSequences(((org.biolegato.gdesupport.formats.DataFormat) openDialog.getFileFilter()).readFile(ofile));
				    } else {
					datamodel.addSequences(
						org.biolegato.gdesupport.formats.DataFormat.auto(ofile));
				    }
				} catch (Throwable e) {
				    e.printStackTrace();
				}
			    }
			}
		    }
		    if (openDialog.getCurrentDirectory() != null) {
			BLMain.setCurrentPWD(openDialog.getCurrentDirectory());
		    }
                }
            }
        }));

        /*******************************
         * Add the "Save As..." button *
         *******************************/
        BLMain.addMenuItem(1, "File", new JMenuItem(new AbstractAction("Save As...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - save the file */
                JFileChooser saveDialog = new JFileChooser();

                saveDialog.setCurrentDirectory(BLMain.getCurrentPWD());
                saveDialog.setAcceptAllFileFilterUsed(false);
                org.biolegato.gdesupport.formats.DataFormat.addFormats(saveDialog,
                        BLMain.getProperty("default.fileformat").toString());
                // if a file is selected, save to it
                if (saveDialog.showSaveDialog(BLMain.getJFrame()) ==
                        JFileChooser.APPROVE_OPTION &&
                        saveDialog.getSelectedFile() != null &&
                        (!saveDialog.getSelectedFile().exists() ||
                        javax.swing.JOptionPane.showConfirmDialog(null,
                        "Overwrite existing file?", "Overwrite", javax.swing.JOptionPane.OK_CANCEL_OPTION,
                        javax.swing.JOptionPane.QUESTION_MESSAGE) != javax.swing.JOptionPane.CANCEL_OPTION)) {
                    // write the file
                    try {
                        ((org.biolegato.gdesupport.formats.DataFormat) saveDialog.getFileFilter()).writeFile(saveDialog.getSelectedFile(), datamodel.toArray());
                    } catch (Throwable e) {
                        e.printStackTrace();
                    }
		    if (saveDialog.getCurrentDirectory() != null) {
			BLMain.setCurrentPWD(saveDialog.getCurrentDirectory());
		    }
                }
            }
        }));
        
        /*******************************
         * Add the "Properties" button *
         *******************************/
        BLMain.addMenuItem(2, "File", new JMenuItem(new AbstractAction("Properties...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
            }  /* Sets the mnemonic for the event */


            public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - open canvas properties window */
                new GDECanvasProperties(BLMain.getJFrame(), BLMain.getProperties());
            }
        }));
        
        // NOTE: count++ increments count after retrieving its value for use in the below functions
        // this choice of count++ allows variable length menus or simple re-ordering of choices
        BLMain.addMenuHeading(1, "Edit").insert(cutMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(copyMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(pasteMenuItem, menuInsertCount++);
        if (currentPane == nameList) {
            BLMain.addMenuHeading(1, "Edit").insert(groupMenuItem, menuInsertCount++);
            BLMain.addMenuHeading(1, "Edit").insert(ungroupMenuItem, menuInsertCount++);
            BLMain.addMenuHeading(1, "Edit").insert(getInfoMenuItem, menuInsertCount++);
        }
        if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
            BLMain.addMenuHeading(1, "Edit").insert(undoMenuItem, menuInsertCount++);
            BLMain.addMenuHeading(1, "Edit").insert(redoMenuItem, menuInsertCount++);
        }
        BLMain.addMenuHeading(1, "Edit").insert(selectGroupMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(selectAllMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(changeCaseMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(selectByNameMenuItem, menuInsertCount++);
        
	// create new objects
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    dataCollector = new UndoableGDETextArea(this);
	} else {
	    dataCollector = new GDETextArea(canvasSelf);
	}
	
        // setup listeners
        dataCollector.addCursorListener(this);
        dataCollector.addModeListener(this);
	nameList.addListSelectionListener(this);
	
	// setup font listener
        BLMain.addPropertiesListener("font.size", this);
        BLMain.addPropertiesListener("font.bold", this);
        
        // add items to the textarea's popup menu
        dataCollector.addPopupMenuItem(splitMenuItem);

        // create the scroll panes
        dataPane = new JScrollPane(dataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        JScrollPane listPane = new JScrollPane(nameList, JScrollPane.VERTICAL_SCROLLBAR_NEVER, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
	dataPane.setPreferredSize(new Dimension(200, 150));
        listPane.setPreferredSize(new Dimension(100, 150));

        // make the vertical scroll bars the same for both scroll panes
        dataPane.setVerticalScrollBar(listPane.getVerticalScrollBar());

        // sets the fixed cell height for the list's items to the row height of the jtextarea
        updateFont();

        // add the dataPane to the canvasPane
        canvasPane.add(dataPane);

        // create the main split pane
        JSplitPane mainPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, listPane, canvasPane);
        mainPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
        mainPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);

        // create the status bar
        Box statusBar = new Box(BoxLayout.LINE_AXIS);
        statusBar.setAlignmentX(Box.LEFT_ALIGNMENT);
        statusBar.add(status);
        statusBar.add(insertStatus);

        // add the panes to the canvas
        add(mainPane, BorderLayout.PAGE_START);
        add(new JSeparator(SwingConstants.HORIZONTAL));
        add(statusBar, BorderLayout.PAGE_END);
    }

    /**
     * Returns the name to display in the canvas tab for
     * @return  "GDE"
     */
    public String getTabName () {
        return "GDE";
    }

    /**
     * Updates the font of the canvas to "newFont".
     */
    private void updateFont () {
        int fontsize = 12;
        String fontsizestr = BLMain.getProperty("font.size");
        if (fontsizestr != null) {
            try {
                fontsize = Integer.parseInt(fontsizestr);
            } catch (NumberFormatException nfe) {
                nfe.printStackTrace();
            }
        }
        Font currentFont = new Font("Lucida Sans Typewriter", Font.PLAIN, fontsize);
        
        if ("true".equalsIgnoreCase(BLMain.getProperty("font.bold"))) {
            dataCollector.setFont(currentFont.deriveFont(Font.BOLD));
        } else {
            dataCollector.setFont(currentFont);
        }
        nameList.setFont(currentFont);
    }

    /**
     * Copies content from the current Editable object to the clipboads.
     */
    public void copy () {
        if (currentPane != null) {
            currentPane.copy();
        }
    }

    /**
     * Cuts content from the current Editable object to the clipboads.
     */
    public void cut () {
        if (currentPane != null) {
            currentPane.cut();
        }
    }

    /**
     * Pastes the current clipboard into the current Editable object.
     */
    public void paste () {
        if (currentPane != null) {
            currentPane.paste();
        }
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    private void changeCase () {
        if (currentPane != null) {
            currentPane.changeCase();
        }
    }

    /**
     * Receives cursor updates for use in the status bar.
     *
     * @param source the source of the cursor change.
     * @param column the new column of the cursor.
     * @param row the new row of the cursor.
     */
    public void cursorChange (BLTextArea source, int column, int row) {
        status.setText("Row: " + (row + 1) + " Col: " + (column + 1));
    }

    /**
     * Receives insertion mode change updates for use in the status bar.
     *
     * @param mode the new insertion mode status.
     */
    public void insertionMode (boolean mode) {
        if (mode) {
            insertStatus.setText("[INS]");
        } else {
            insertStatus.setText("     ");
        }
    }

    /**
     * Used to intercept font size changes (since the font size is stored in the properties)
     *
     * @param key the key of the property changed.
     * @param value the new value of the property.
     */
    public void propertiesUpdate(String key, String value) {
        if (key.toLowerCase().startsWith("font.")) {
            updateFont();
        }
    }

    /**
     * Clears the data collector when an item is selected from the sequence list
     * @param e currently ignored
     */
    public void valueChanged(ListSelectionEvent e) {
        if (currentPane != nameList) {
            currentPane = nameList;
            dataCollector.clearSelection();
            if (altDataCollector != null) {
                altDataCollector.clearSelection();
            }
            BLMain.addMenuHeading(1, "Edit").insert(groupMenuItem, 3);
            BLMain.addMenuHeading(1, "Edit").insert(ungroupMenuItem, 4);
            BLMain.addMenuHeading(1, "Edit").insert(getInfoMenuItem, 5);
        }
    }
    
    /**
     * Manages mutual exclusion between the textarea and the list
     *
     * @param source the source of the event
     */
    public void selectionMade (GDETextArea source) {
        if (currentPane != source) {
            currentPane = source;
            nameList.clearSelection();
            if (altDataCollector != null && source != altDataCollector) {
                altDataCollector.clearSelection();
            }
            if (source != dataCollector) {
                dataCollector.clearSelection();
            }
            BLMain.addMenuHeading(1, "Edit").remove(groupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(ungroupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(getInfoMenuItem);
        }
    }

/////////////////////////
//*********************//
//* UNDO/REDO METHODS *//
//*********************//
/////////////////////////
    /**
     * Undoes data modification.
     *
     * @return whether or not the undo was successful.
     */
    public boolean undo () {
        boolean result =  ! undoStack.empty();  // the result of the undo.
        Undoable undone = null;                 // the redo object (to redo the operation).

        if (result) {
            undone = undoStack.pop().undo();
            if (undone != null) {
                redoStack.push(undone);
            }
        }
        return result;
    }

    /**
     * Redoes data modification.
     *
     * @return whether or not the redo was successful.
     */
    public boolean redo () {
        boolean result =  ! redoStack.empty();  // the result of the undo.
        Undoable redone = null;                 // the undo object (to undo the redone operation).

        if (result) {
            redone = redoStack.pop().undo();
            if (redone != null) {
                undoStack.push(redone);
            }
        }
        return result;
    }
    
    /**
     * Adds an undoable object to the undo stack.
     *
     * @param undo the object to add to the stack.
     */
    public void addUndo (Undoable undo) {
	int undomax = 0;
	
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    redoStack.clear();
	    undoStack.push(undo);

	    if (BLMain.getProperty("undo.size") != null && BLMain.testNumber(BLMain.getProperty("undo.size").toCharArray())) {
		try {
		    undomax = Integer.parseInt(BLMain.getProperty("undo.size"));
		    while (undomax > 0 && undoStack.size() > undomax) {
			undoStack.remove(0);
		    }
		} catch (Throwable nfe) {
		}
	    }
	}
    }

///////////////////////////
//***********************//
//* CLIPBOARD FUNCTIONS *//
//***********************//
///////////////////////////
    /**
     * Obtains the current contents of the clipboard (null if empty).
     *
     * @return the current contents of the clipboard.
     */
    public static Cell[] getClipboard() {
        String string = "";
        Cell[] clipboard = null;
        Transferable t = Toolkit.getDefaultToolkit().getSystemClipboard().getContents(null);

        try {
            if (t != null) {
                if (t.isDataFlavorSupported(DataFlavor.stringFlavor)) {
                    clipboard = DataFormat.auto(new BufferedReader(new StringReader((String)
                            t.getTransferData(DataFlavor.stringFlavor))));
                }
            }
        } catch (UnsupportedFlavorException e) {
        } catch (IOException e) {
        }
        return clipboard;
    }

    /**
     * Changes the current contents of the clipboard.
     *
     * @param copy the new content for the clipboard.
     */
    public static void setClipboard(Cell[] copy) {
        Cell[] cells = new Cell[copy.length];
        String result = "";
        for (int count = 0; count < copy.length; count++) {
            cells[count] = new Cell(copy[count]);
        }
        for (Cell c : cells) {
            result += DataFormat.GENBANK.translateTo(c);
        }
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(result), null);
    }

    /**
     * Passes the current datamodel to any GDE canvas related components
     *
     * @returns the GDE canvas sequence data model
     */
    public GDEModel getDataModel() {
        return datamodel;
    }

    /**
     * Returns the current/selected data in the canvas.
     *
     * @return the current data for usage by commands
     */
    public Cell[] getData () {
        Cell[] result = null;

        // check whether to use the row(s) selected or the selection in the JTextArea
        if (currentPane != null) {
            // get the text selected
            result = currentPane.getData();
        }
        return (result != null ? result : new Cell[0]);
    }

    /**
     * Reads a file into the canvas
     **
     * @param format the file format to use for parsing the file.
     * @param currentFile the file to read in.
     */
    public void readFile(String format, File currentFile) {
        DataFormat formatUsed = DataFormat.getFormat(format);
        
        try {
            if (formatUsed != null) {
                datamodel.addSequences(formatUsed.readFile(currentFile));
            } else if (format == null || "".equals(format)) {
                // TODO: Improve
                datamodel.addSequences(DataFormat.auto(currentFile));
            } else {
                BLMain.error("Unsupported file format", "GDECanvas");
            }
        } catch (Throwable th) {
            th.printStackTrace();
        }
    }

    /**
     * Writes a file out from the canvas
     **
     * @param format the file format to use for writing the file.
     * @param currentFile the file to write out.
     */
    public void writeFile(String format, File currentFile) {
        DataFormat formatUsed = DataFormat.getFormat(format);
        
        try {
            if (formatUsed != null) {
                formatUsed.writeFile(currentFile, getData());
            } else {
                BLMain.error("Unsupported file format", "GDECanvas");
            }
        } catch (Throwable th) {
            th.printStackTrace();
        }
    }
}
