/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.colourmask;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Hashtable;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.colourmap.ColourMap;
import org.biolegato.gdesupport.canvas.colourmap.MonochromeColourMap;

/**
 * The menu item to handle importing colour masks
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class ImportFileAction extends Box implements ActionListener {

    /**
     * The window to associate the import with.
     */
    private JDialog window;
    /**
     * The combobox to use for selecting imported colour masks from.
     */
    private JComboBox combobox;
    /**
     * The list of all colour masks loaded in the system.
     */
    protected static Hashtable<String, ColourMap> colourMasks = new Hashtable<String, ColourMap>();
    /**
     * Used for serialization purposes.
     */
    private static final long serialVersionUID = 7526472295622777037L;
    /**
     * The colour mask file reader for GDE's position-based colour mask files.
     */
    private static final CharacterColourMaskReader characterMaskFile = new CharacterColourMaskReader();
    /**
     * The colour mask file reader for BioLegato's character-based colour mask files.
     */
    private static final GDEColourMaskFile gdeMaskFile = new GDEColourMaskFile();
    /**
     * The default colour mask for BioLegato
     */
    private static ColourMap defaultMask = new MonochromeColourMap(Color.BLACK, Color.WHITE);

    /**
     * Load the default colour masks (this was the only way I was able to get this code to work properly).
     */
    static {
        if (colourMasks.isEmpty()) {
            if (System.getenv("BL_MASKS") != null && !"".equals(System.getenv("BL_MASKS"))) {
                load(new File(BLMain.envreplace(System.getenv("BL_MASKS"))));
            } else if (BLMain.getProperty("seqcolourdir") != null && !"".equals(BLMain.getProperty("seqcolourdir"))) {
                load(new File(BLMain.envreplace(BLMain.getProperty("seqcolourdir"))));
            }
	    if (colourMasks.containsKey("default")) {
		defaultMask = colourMasks.get("default");
	    }
        }
    }

    /**
	 * Creates a new ImportFileAction object.
	 * 
	 * @param window the window to associate the import with.
	 * @param selectedMask the default mask for the combobox.
	 */
    public ImportFileAction(JDialog window, final Object selectedMask) {
        super(BoxLayout.LINE_AXIS);

        // set class parameters
        this.combobox = new JComboBox(colourMasks.values().toArray());
        this.window = window;

        // add the currently loaded masks combobox
	if (!colourMasks.containsKey("default")) {
	    combobox.insertItemAt(defaultMask, 0);
	}
	if (selectedMask == null || "".equals(selectedMask) || "<default>".equals(selectedMask)) {
	    combobox.setSelectedItem(defaultMask);
	} else {
	    if (!colourMasks.containsValue(selectedMask)) {
		combobox.insertItemAt(selectedMask, 0);
	    }
	    combobox.setSelectedItem(selectedMask);
	}
        add(combobox);

        // add the import file button
        JButton importButton = new JButton("Import file...");
        importButton.addActionListener(this);
        add(importButton);
    }

    /**
     * Performs all of the import file functionality.
     *
     * @param e is ignored by this function.
     */
    public void actionPerformed(ActionEvent e) {
        JFileChooser openDialog = new JFileChooser();   // the JFileChooser for selecting the file to import.
        ColourMap[] importMasks = null;                 // the colour masks that were imported.

        // set the parameters for the open file dialog box
        openDialog.setCurrentDirectory(BLMain.getCurrentPWD());
        openDialog.setAcceptAllFileFilterUsed(false);

        // add the default file filters
        openDialog.addChoosableFileFilter(characterMaskFile);
        openDialog.addChoosableFileFilter(gdeMaskFile);

        // set the default file filter
        openDialog.setFileFilter(characterMaskFile);

        // if a file is selected, open it
        if (openDialog.showOpenDialog(window) == JFileChooser.APPROVE_OPTION) {
	    if (openDialog.getSelectedFile() != null && openDialog.getSelectedFile().exists() && openDialog.getSelectedFile().isFile()) {
		if (openDialog.getFileFilter() != null && openDialog.getFileFilter() instanceof ColourMaskReader) {
		    addMasks(((ColourMaskReader) openDialog.getFileFilter()).readColourMapFile(openDialog.getSelectedFile()), this.combobox);
		}
	    }
	    if (openDialog.getCurrentDirectory() != null) {
		BLMain.setCurrentPWD(openDialog.getCurrentDirectory());
	    }
        }
    }

    /**
     * Returns the default colour mask
     */
    public static ColourMap getDefaultMap() {
	return defaultMask;
    }
    
    /**
     * Loads colour masks into biolegato.
     */
    public static void addMasks(ColourMap[] importMasks, JComboBox addCombobox) {
        int number = 0;                                 // number to append to duplicate masks

        for (ColourMap mask : importMasks) {
            if (colourMasks.contains(mask.getName())) {
                while (colourMasks.contains(mask.getName() + " (" + number + ")")) {
                    number++;
                }
                mask.setName(mask.getName() + " (" + number + ")");
            }
            if (addCombobox != null) {
                addCombobox.addItem(mask);
            }
            colourMasks.put(mask.getName(), mask);
        }
    }

    /**
     * Returns the current selected colour mask for the sequence.
     *
     * @return the currently selected colour mask for the sequence.
     */
    public Object getSelectedItem() {
        return combobox.getSelectedItem();
    }

    /**
     * Loads all of the colour masks contained in a file or directory.
     *
     * @param location the file object to read colour masks from.
     */
    public static void load(File location) {
        if (location.exists() && location.canRead()) {
            if (location.isDirectory()) {
                for (File file : location.listFiles()) {
                    load(file);
                }
            } else if (location.isFile() && location.getAbsolutePath().toLowerCase().endsWith(".csv")) {
                ImportFileAction.addMasks(new CharacterColourMaskReader().readColourMapFile(location), null);
            }
        }
    }
}
