/*
 * Dataset.java
 *
 * Created on September 30, 2008, 11:08 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.data;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.gdesupport.canvas.data.CellListener;
import java.util.LinkedList;
import org.biolegato.core.main.BLMain;

/**
 * The internal document format for BioLegato.
 * <p>
 * This document is structured as a linked list of sequences.  Each character has an offset based on its
 * position within the list and it's position within its containing sequence.  Sequences start at 0
 * (first character in the first sequence in the list, and end with the last character in the last sequence
 * within the list.
 * </p>
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class Dataset implements CellListener, Transferable {

    /**
     * This linked list used to store all of the lines in the document.
     * <p>
     *  Each line is stored as a linked list of sequence wrappers.
     *  Each sequence wrapper is characterized by a sequence and a start offset.
     *  The start offset is used to determine the offset of the sequence within
     *  the document.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private List<List<Cell>> data = new ArrayList<List<Cell>>();
    private final List<Cell> lines = new ArrayList<Cell>();
    /**
     * This list is used to store all of the document's sequence listeners.
     * <p>
     *  Each time any modification is done to a sequence within the document, or
     *  any sequences are added/removed from the document, all of the document's
     *  listeners are notified of the event.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final List<DatasetListener> listeners = new LinkedList<DatasetListener>();
    /**
     * A data flavour representing the Dataset data type.
     */
    public static final DataFlavor seqDocFlavour = new DataFlavor(Dataset.class, "BioLegato sequence document");

    /**
     * Creates a new instance of Dataset
     */
    public Dataset() {
    }

    /**
     * Creates a new instance of Dataset using the existing sequence list
     * 
     * 
     * 
     * @param sequenceList the list of sequences to initialize the SDatasetwith.
     */
    public Dataset(Cell[] row) {
        for (Cell data : row) {
            addRow(getRowCount(), data);
        }
    }

////////////////////////////////
//****************************//
//* DIRECT DATA MODIFICATION *//
//****************************//
////////////////////////////////
    /**
     * Adds a sequence to the data container.  This function calls all listeners.
     **
     * @param y is the line number to insert the sequence.
     * @param seq is the sequence to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean addRow(int y, Cell seq) {
        boolean result = false;

        // TODO: add x support
        if (y >= 0 && y <= getRowCount()) {
            if (seq != null) {
                lines.add(y, seq);
                seq.addListener(this);
                for (DatasetListener listener : listeners) {
                    listener.sequenceAdded(this, y, seq);
                }
            }
            result = true;
        }
        return result;
    }

    /**
     * Removes an array of sequences from the data container.
     **
     * @param lineNumbers the line numbers to remove.
     */
    public void removeRows(final int[] lineNumbers) {
        int length;			    // the length of the sequence removed.
        String text;			    // the data of the sequence removed.
        Cell removed;		    // the sequence wrapper object removed.

        // sort the line numbers
        Arrays.sort(lineNumbers);

        // ensure that there are lines to delete
        if (!lines.isEmpty()) {
            // itterate backwards through each sequence line number and delete it (using the removeSequence method)
            for (int count = lineNumbers.length - 1; count >= 0; count--) {
                // check the bounds of the line number to delete
                if (getRowCount() > lineNumbers[count] && lineNumbers[count] >= 0) {
                    // remove the sequence
                    removed = lines.remove(lineNumbers[count]);

                    // get information about the sequence removed
                    text = (String) removed.get("sequence");
                    length = text.length();
                    removed.removeListener(this);

                    // send sequence removed event
                    for (DatasetListener listener : listeners) {
                        listener.sequenceRemoved(this, lineNumbers[count], removed);
                    }
                }
            }
        }
    }
//////////////////////
//******************//
//* DATA RETRIEVAL *//
//******************//
//////////////////////
    /**
     * Retrieves a sequence object from the Dataset specified by its line number.
     * 
     * 
     * 
     * @param y the line number to retreive the sequence.
     * @return the sequence.
     */
    public Cell get(final int y) {
        return (y >= 0 && y < getRowCount() ? lines.get(y) : null);
    }

/////////////////
//*************//
//* LISTENERS *//
//*************//
/////////////////
    /**
     * Adds a listener object to the data container.
     *
     * @param listener the listener to add.
     */
    public void addListener(final DatasetListener listener) {
        listeners.add(listener);
    }

    /**
     * Adds a listener object to the data container.
     *
     * @param listener the listener to add.
     */
    public void removeListener(final DatasetListener listener) {
        listeners.remove(listener);
    }
    
    /**
     * Called when a field in a sequence is modified.
     *
     * @param sequence the sequence modified.
     * @param key the key of the modified field in the sequence.
     */
    public void sequenceChanged(final Cell sequence, final String key) {
        for (DatasetListener listener : listeners) {
            listener.sequenceChanged(this, lines.indexOf(sequence), sequence, key);
        }
    }

////////////////////
//****************//
//* TRANSFERABLE *//
//****************//
////////////////////
    /**
     * Dictates what formats the Dataset can be converted to.
     * 
     * 
     * 
     * @return an array of supported formats
     */
    public DataFlavor[] getTransferDataFlavors() {
        return new DataFlavor[]{seqDocFlavour, DataFlavor.stringFlavor};
    }

    /**
     * Dictates whether a given format is supported for conversion
     *
     * @param flavour the flavour to test for compatability
     * @return true if the format is supported
     */
    public boolean isDataFlavorSupported(DataFlavor flavour) {
        boolean result = false;
        DataFlavor[] test = getTransferDataFlavors();

        for (int count = 0; count < test.length && !result; count++) {
            result = flavour.equals(test[count]);
        }
        return result;
    }

    /**
     * Translates the Dataset to the given format
     * 
     * 
     * 
     * @param flavour the data flavour to use for translation
     * @return the translated object
     * @throws UnsupportedFlavorException if the SeqDatasetnnot be translated into the requested data flavour
     * @throws IOException if there is a problem with I/O during the translation
     */
    public Object getTransferData(DataFlavor flavour) throws UnsupportedFlavorException, IOException {
        Object result = this;

        if (!isDataFlavorSupported(flavour)) {
            throw new UnsupportedFlavorException(flavour);
        }
        if (DataFlavor.stringFlavor.equals(flavour)) {
            result = toString();
        }
        return result;
    }

///////////////
//***********//
//* GENERAL *//
//***********//
///////////////
    /**
     * Converts the Dataset into a string representation
     **
     * @return the string representation of the sequence document
     */
    @Override
    public String toString() {
        String result = "";
        for (Cell seq : lines) {
            if (!result.equals("")) {
                result += "\n";
            }
            result += seq.get("sequence");
        }
        return result;
    }

    /**
     * Converts the Dataset into a sequence array representation
     **
     * @return the sequence array representation of the sequence document
     */
    public Cell[] toArray() {
        return lines.toArray(new Cell[0]);
    }

    /**
     * Finalizes the object - removes all references to itself
     */
    @Override
    protected void finalize() {
        for (Cell seq : lines) {
            seq.removeListener(this);
        }
        lines.clear();
    }

    /**
     * Returns the number of lines in the document.
     **
     * @return the number of lines in the document.
     */
    public int getRowCount() {
        return lines.size();
    }
}
