/*
 * Dataset.java
 *
 * Created on September 30, 2008, 11:08 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.data;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.gdesupport.canvas.data.CellListener;
import java.util.LinkedList;
import org.biolegato.core.main.BLMain;

/**
 * The internal document format for BioLegato.
 * <p>
 * This document is structured as a linked list of sequences.  Each character has an offset based on its
 * position within the list and it's position within its containing sequence.  Sequences start at 0
 * (first character in the first sequence in the list, and end with the last character in the last sequence
 * within the list.
 * </p>
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEModel implements CellListener, Transferable {

    /**
     * This linked list used to store all of the lines in the document.
     * <p>
     *  Each line is stored as a linked list of sequence wrappers.
     *  Each sequence wrapper is characterized by a sequence and a start offset.
     *  The start offset is used to determine the offset of the sequence within
     *  the document.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final List<Cell> lines = new ArrayList<Cell>();
    /**
     * A data flavour representing the Dataset data type.
     */
    public static final DataFlavor seqDocFlavour = new DataFlavor(GDEModel.class, "BioLegato sequence document");
    /**
     * Keeps track of the maximum group number in usage.
     */
    private int maxGroup = 1;
    /**
     * A hashtable used to map all groups to sequences 
     */
    private final Hashtable<Integer, List<Integer>> group2seq = new Hashtable<Integer, List<Integer>>();
    /**
     * This list is used to store all of the document's sequence listeners.
     * <p>
     *  Each time any modification is done to a sequence within the document, or
     *  any sequences are added/removed from the document, all of the document's
     *  listeners are notified of the event.
     * </p>
     *
     * @see org.biolegato.core.data.sequence.Cell
     */
    private final List<GDEModelListener> listeners = new LinkedList<GDEModelListener>();

    /**
     * Creates a new instance of Dataset
     */
    public GDEModel() {
    }

////////////////////////////////
//****************************//
//* DIRECT DATA MODIFICATION *//
//****************************//
////////////////////////////////
    /**
     * Adds a sequence to the data container.  This function calls all listeners.
     **
     * @param lineNumber is the line number to insert the sequence.
     * @param seq is the sequence to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean addSequence(int y, Cell seq) {
        boolean result = false;

        // TODO: add x support
        if (y >= 0 && y <= getLineCount()) {
            if (seq != null) {
                lines.add(y, seq);
                seq.addListener(this);
                for (GDEModelListener listener : listeners) {
                    listener.sequenceAdded(this, y, seq);
                }
            }
            result = true;
        }
        return result;
    }

    /**
     * Removes an array of sequences from the data container.
     **
     * @param lineNumbers the line numbers to remove.
     */
    public void removeSequences(final int[] lineNumbers) {
        int length;			    // the length of the sequence removed.
        String text;			    // the data of the sequence removed.
        Cell removed;		    // the sequence wrapper object removed.

        // sort the line numbers
        Arrays.sort(lineNumbers);

        // ensure that there are lines to delete
        if (!lines.isEmpty() && lineNumbers != null) {
            // itterate backwards through each sequence line number and delete it (using the removeSequence method)
            for (int count = lineNumbers.length - 1; count >= 0; count--) {
                // check the bounds of the line number to delete
                if (getLineCount() > lineNumbers[count] && lineNumbers[count] >= 0) {
                    // remove the sequence
                    removed = lines.remove(lineNumbers[count]);

                    // get information about the sequence removed
                    text = (String) removed.get("sequence");
                    length = text.length();
                    removed.removeListener(this);

                    // send sequence removed event
                    for (GDEModelListener listener : listeners) {
                        listener.sequenceRemoved(this, lineNumbers[count], removed);
                    }
                }
            }
        }
    }

//////////////////////////////////
//******************************//
//* INDIRECT DATA MODIFICATION *//
//******************************//
//////////////////////////////////
    /**
     * Inserts a string into the document on a given line.
     **
     * @param col the offset in the object in the dataset to insert the text.
     * @param x the x-co-ordinate to insert the string.
     * @param line the line-co-ordinate to insert the string.
     * @param string the text to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean insert(final int col, final int line, final String string) {
        boolean result = (string == null || "".equals(string));

        if (!result && line >= 0 && line < getLineCount() && col >= 0 && col <= getLineLength(line)) {
            result = get(line).insertField("sequence", col, string);
        }
        return result;
    }

    /**
     * Removes text from the document.
     * <p>
     *  This method will delete line endings and sequences as well as individual characters from the
     *  document.
     * </p>
     * 
     * <b>NOTE: ALL DELETIONS ARE PERFORMED AS FOLLOWS:</b>
     * <br />
     *  Cell[0]: ggggaaaa<br />
     *          ....<br />
     *  Cell[3]: cccctttt
     * <br />
     * 	<i>delete(4, 0, 4, 3);</i>
     * <br />
     *  Result: gggg
     *          ....
     *          cccc<br />
     * 
     * @param col the X-offset/column number to start the deletion from.
     * @param line the Y-offset/line number to delete characters from.
     * @param cols the width of the deletion (measured in characters along the X-axis).
     * @return true if the deletion was successful, otherwise false.
     */
    public boolean delete(final int col, final int line, final int cols) {
        String data;
        boolean result = (getLineCount() == 0 && line == 0 && col == 0 && cols == 0);

        if (line >= 0 && line < getLineCount()) {
            data = getText(line);
            if (col >= 0 && cols >= 0 && col + cols <= data.length()) {
                if (col + cols >= data.length()) {
                    data = data.substring(0, col);
                } else {
                    data = data.substring(0, col) + data.substring(col + cols);
                }
                get(line).put("sequence", data);
                result = true;
            } else if (col == 0 && cols == 0 && data.length() == 0) {
                result = true;
            }
        }

        return result;
    }

    /**
     * Removes text from the document.
     * <p>
     *  This method will delete line endings and sequences as well as individual characters from the
     *  document.
     * </p>
     * 
     * <b>NOTE: ALL DELETIONS ARE PERFORMED AS FOLLOWS:</b>
     * <br />
     *  Cell[0]: ggggaaaa<br />
     *          ....<br />
     *  Cell[3]: cccctttt
     * <br />
     * 	<i>delete(4, 0, 4, 3);</i>
     * <br />
     *  Result: gggg
     *          ....
     *          cccc<br />
     * 
     * @param col the X-offset/column number to start the deletion from.
     * @param line the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param lines the height of the deletion (measured in sequences along the Y-axis).
     * @return true if the deletion was successful, otherwise false.
     */
    public boolean delete(final int col, final int line, final int cols, final int lines) {
        String data = "";                                       // the data for the sequence after the removal
        boolean result = false;		                        // the result of the operation (true if successful)

        // make sure all of the parameters are valid
        if (getLineCount() > 0
                && line >= 0 && line + lines < getLineCount() && col >= 0) {
            // substring the sequence (delete the data)
            for (int count = 0; count <= lines; count++) {
                result |= delete(col, line + count, cols);
            }
        }
        return result;
    }
//////////////////////
//******************//
//* DATA RETRIEVAL *//
//******************//
//////////////////////
    /**
     * Retrieves a sequence object from the Dataset specified by its line line.
     **
     * @param number the line line to retreive the sequence.
     * @return the sequence.
     */
    public Cell get(final int number) {
        return (number >= 0 && number < getLineCount() ? lines.get(number) : null);
    }

    /**
     * Retrieves text from the Dataset specified by its line line.
     **
     * @param number the line number to retreive the text from.
     * @return the text.
     */
    private String getText(int number) {
        Cell seq = get(number);
        return (seq != null ? (String) seq.get("sequence") : "");
    }

    /**
     * Returns the number of lines in the document.
     **
     * @return the number of lines in the document.
     */
    public int getLineCount() {
        return lines.size();
    }
    
    /**
     * Retrieves the length of a line in the document.
     **
     * @param line the line line to find the length of.
     * @return the length of the line (in characters).
     */
    public int getLineLength(final int line) {
        return getText(line).length();
    }

    /**
     * Returns the length of the longest line in the data container
     **
     * @return the length of the longest line
     */
    public int getLongestLine() {
        int lineLength = 0;

        for (int count = 0; count < getLineCount(); count++) {
            lineLength = Math.max(lineLength, getLineLength(count));
        }
        return lineLength;
    }

/////////////////
//*************//
//* LISTENERS *//
//*************//
/////////////////
    /**
     * Adds a listener object to the data container.
     *
     * @param listener the listener to add.
     */
    public void addListener(final GDEModelListener listener) {
        listeners.add(listener);
    }

    /**
     * Adds a listener object to the data container.
     **
     * @param listener the listener to add.
     */
    public void removeListener(final GDEModelListener listener) {
        listeners.remove(listener);
    }
    
    /**
     * Called when a field in a sequence is modified.
     **
     * @param sequence the sequence modified.
     * @param key the key of the modified field in the sequence.
     */
    public void sequenceChanged(final Cell sequence, final String key) {
        for (GDEModelListener listener : listeners) {
            listener.sequenceChanged(this, lines.indexOf(sequence), sequence, key);
        }
    }

    
////////////////////
//****************//
//* TRANSFERABLE *//
//****************//
////////////////////
    /**
     * Dictates what formats the Dataset can be converted to.
     **
     * @return an array of supported formats
     */
    public DataFlavor[] getTransferDataFlavors() {
        return new DataFlavor[]{seqDocFlavour, DataFlavor.stringFlavor};
    }

    /**
     * Dictates whether a given format is supported for conversion
     **
     * @param flavour the flavour to test for compatability
     * @return true if the format is supported
     */
    public boolean isDataFlavorSupported(DataFlavor flavour) {
        boolean result = false;
        DataFlavor[] test = getTransferDataFlavors();

        for (int count = 0; count < test.length && !result; count++) {
            result = flavour.equals(test[count]);
        }
        return result;
    }

    /**
     * Translates the Dataset to the given format
     **
     * @param flavour the data flavour to use for translation
     * @return the translated object
     * @throws UnsupportedFlavorException if the SeqDatasetnnot be translated into the requested data flavour
     * @throws IOException if there is a problem with I/O during the translation
     */
    public Object getTransferData(DataFlavor flavour) throws UnsupportedFlavorException, IOException {
        Object result = this;

        if (!isDataFlavorSupported(flavour)) {
            throw new UnsupportedFlavorException(flavour);
        }
        if (DataFlavor.stringFlavor.equals(flavour)) {
            result = toString();
        }
        return result;
    }

///////////////
//***********//
//* GENERAL *//
//***********//
///////////////
    /**
     * Finalizes the object - removes all references to itself
     */
    @Override
    protected void finalize() {
        for (Cell seq : lines) {
            seq.removeListener(this);
        }
        lines.clear();
    }

    private void setgroup(Integer groupNumber, Cell sequence) {
    }

    public void group (int[] sequences) {
        Cell seq = null;
	List<Cell> group = null;
	Integer groupNumber = maxGroup;
        
        ungroup(sequences);
	
	for (int number : sequences) {
            // find the sequence to process
            seq = get(number);
            
            // ensure that we don't process invalid sequence numbers
            if (seq != null) {
                // create new group reference
                seq.put("group", groupNumber);

                // ensure that there is an entry in group2seq to add the sequence to.
                if (!group2seq.containsKey((Integer) groupNumber)) {
                    group2seq.put((Integer) groupNumber, new ArrayList<Integer>());
                }

                group2seq.get(groupNumber).add(number);
                seq.put("group", groupNumber);
            }
	}
	maxGroup++;
    }
    
    public void ungroup (int[] sequences) {
        Cell seq = null;
	List<Integer> group = null;
	Integer groupNumber = null;
        
	for (int number : sequences) {
            seq = get(number);
            
            // remove previous group reference
            if (seq != null && seq.containsKey("group") && seq.get("group") instanceof Integer) {
                groupNumber = (Integer) seq.get("group");
                group = getgroup(groupNumber);
                group.remove(number);
                
                if (group.isEmpty()) {
                    while (maxGroup > 1 && maxGroup < group2seq.size() && group2seq.get(maxGroup - 1).isEmpty()) {
                        maxGroup--;
                    }
                    group2seq.remove(groupNumber);
                }

                seq.remove("group");
            }
	}
    }
    
    public List<Integer> getgroup(Integer groupNumber) {
	List<Integer> group = null;
	
	if (groupNumber != null && group2seq.containsKey(groupNumber)) {
	    group = group2seq.get(groupNumber);
	} else {
	    group = new LinkedList<Integer>();
	}
	return group;
    }

    public void addSequences(Cell[] list) {
        int position = getLineCount();
        
        if (list != null) {
            for (Cell entry : list) {
                addSequence(position, entry);
                position++;
            }
        }
    }

    public Cell[] toArray() {
        return lines.toArray(new Cell[0]);
    }
}
