/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.list;

import java.awt.Color;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.*;
import org.biolegato.gdesupport.canvas.data.GDEModel;
import org.biolegato.gdesupport.canvas.undo.UndoMulti;
import org.biolegato.gdesupport.canvas.undo.UndoRowDeletion;
import org.biolegato.gdesupport.canvas.undo.Undoable;

/**
 * The JList of sequences in the GDE canvas.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEList extends JList implements MouseListener, GDECanvasObject {
    /**
     * The last time the mouse was clicked (used for double click detection)
     */
    private long lastClickTime = -1000;
    /**
     * The entry selected on the last click
     */
    private int lastClickEntry = -1;
    /**
     * Self reference.
     */
    private final GDEList listSelf = this;
    /**
     * Stores a symbolic link to the parent canvas.
     */
    private GDECanvas canvas = null;
    /**
     * The right click menu for the GDE sequence list.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * Used for the relationship between the data model and the list
     */
    protected GDEModel datamodel;
    /**
     * The menu item "Select group"
     */
    public final AbstractAction selectGroupAction = new AbstractAction("Select group") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                selectGroup();
            }

        };
    /**
     * The menu item "Get info..."
     */
    public final AbstractAction getInfoAction = new AbstractAction("Get info...") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_G));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                getInfo();
            }

        };
    /**
     * The menu item "Delete"
     */
    private final JMenuItem deleteMenuItem = new JMenuItem(new AbstractAction("Delete") {

            private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_D));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                delete();
            }

        });
    /**
     * The menu item "Cut"
     */
    private final JMenuItem cutMenuItem = new JMenuItem(new AbstractAction("Cut") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_C));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.cut();
            }

        });
    /**
     * The menu item "Copy"
     */
    private final JMenuItem copyMenuItem = new JMenuItem(new AbstractAction("Copy") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_Y));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.copy();
            }

        });
    /**
     * The menu item "Paste"
     */
    private final JMenuItem pasteMenuItem = new JMenuItem(new AbstractAction("Paste") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.paste();
            }

        });
    /**
     * The menu item "Select All"
     */
    public final AbstractAction selectAllAction = new AbstractAction("Select All") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_A));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
                listSelf.selectAll();
		listSelf.requestFocus();
            }

        };
    /**
     * The abstract action "Select sequence by name"
     */
    public final AbstractAction selectByNameAction = new AbstractAction("Select sequence by name") {

            private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/

            {
                putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
            }  /* Sets the mnemonic for the event */

            public void actionPerformed (java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
		listSelf.requestFocus();
		listSelf.requestFocusInWindow();
		new SelectByNameWindow(BLMain.getJFrame(), listSelf);
            }

        };
    /**
     * Action for grouping sequences in the canvas
     */
    public final AbstractAction groupAction = new AbstractAction("Group") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    datamodel.group(getSelectedIndices());
        }

    };
    /**
     * Action for ungrouping sequences in the canvas
     */
    public final AbstractAction ungroupAction = new AbstractAction("Ungroup") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed (ActionEvent e) {
	    datamodel.ungroup(getSelectedIndices());
        }

    };
    /**
     * Used in serialization.
     */
    private static final long serialVersionUID = 7526472295622777009L;

    /**
     * Constructs a new GDEList
     */
    public GDEList (GDECanvas canvas) {
        super(new SequenceListModel(canvas.getDataModel()));

        // initialize variables
	this.canvas = canvas;
        this.datamodel = canvas.getDataModel();

        // add listeners
        addMouseListener(this);

        // build list popupmenu
        addPopupMenuItem(cutMenuItem);
        addPopupMenuItem(copyMenuItem);
        addPopupMenuItem(pasteMenuItem);
        addPopupMenuItem(deleteMenuItem);
	addPopupMenuItem(new JMenuItem(groupAction));
	addPopupMenuItem(new JMenuItem(ungroupAction));
        addPopupMenuItem(new JMenuItem(selectAllAction));
        addPopupMenuItem(new JMenuItem(selectGroupAction));
	addPopupMenuItem(new JMenuItem(selectByNameAction));
        addPopupMenuItem(new JMenuItem(getInfoAction));

        // update the list content
        //updateList();
        
        // set display properties
        setForeground(Color.BLACK);
        setBackground(new Color(255, 255, 240));
        setSize(getPreferredSize());
    }

    /**
     * Checks for double clicks.  On a double click, this method opens
     * up a sequence properties window.
     *
     * @param event the mouse event to check for the double click.
     */
    public void mouseClicked (MouseEvent event) {
	// if double click select the entire group
        if (event.getClickCount() > 2 || (getSelectedIndex() == lastClickEntry
                && event.getWhen() - lastClickTime < BLMain.DOUBLE_CLICK_TIME)) {
	    selectGroup();
        }
        lastClickTime = event.getWhen();
        lastClickEntry = getSelectedIndex();
    }

    /**
     * Checks for right clicks.  On a right click, this method opens up a popup menu
     *
     * @param event the mouse event to check for the right click.
     */
    public void mousePressed (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX() - getX(), event.getY() - getY());
        }
    }

    /**
     * Checks for right clicks.  On a right click, this method opens up a popup menu
     *
     * @param event the mouse event to check for the right click.
     */
    public void mouseReleased (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX() - getX(), event.getY() - getY());
        }
    }

    /**
     * Currently does nothing
     *
     * @param event ignored.
     */
    public void mouseEntered (MouseEvent event) {
    }

    /**
     * Currently does nothing
     *
     * @param event ignored.
     */
    public void mouseExited (MouseEvent event) {
    }

    /**
     * Converts a row number into a Y co-ordinate
     *
     * @param r the row number
     * @return the corresponding Y value
     */
    public int row2Y (int r) {
        return (r * getFontMetrics(getFont()).getHeight());
    }

    /**
     * Converts a column number into a X co-ordinate
     *
     * @param c the column number
     * @return the corresponding X value
     */
    public int column2X (int c) {
        return (c * getFontMetrics(getFont()).charWidth('a'));
    }

    /**
     * Converts a Y co-ordinate into a row number
     *
     * @param Y the Y value
     * @return the corresponding row number
     */
    public int Y2Row (int Y) {
        return (Y / getFontMetrics(getFont()).getHeight());
    }

    /**
     * Converts a X co-ordinate into a column number
     *
     * @param X the X value
     * @return the corresponding column number
     */
    public int X2Column (int X) {
        return (X / getFontMetrics(getFont()).charWidth('a'));
    }

    /**
     * Copies content from the current Editable object.
     */
    public void copy() {
        GDECanvas.setClipboard(getData());
    }

    /**
     * Cuts content from the current Editable object.
     */
    public void cut() {
        copy();
        delete();
    }

    /**
     * Pastes content into the current Editable object.
     */
    public void paste() {
        int row = datamodel.getLineCount();
	Cell[] clipboard = GDECanvas.getClipboard();

        if (getSelectedIndex() >= 0) {
            row = getSelectedIndex() + 1;
        }
	for (Cell seq : clipboard) {
	    canvas.getDataModel().addSequence(row, seq);
	    row++;
	}
    }

    /**
     * Selects all the sequences within the list.
     */
    public void selectAll () {
        setSelectionInterval(0, getModel().getSize() - 1);
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase () {
        String data = null;
        Cell seq = null;

        for (int sequenceNumber : getSelectedIndices()) {
            seq = datamodel.get(sequenceNumber);
            data = (String) seq.get("sequence");

            if (data.length() >= 0) {
                if (Character.isUpperCase(data.charAt(0))) {
                    seq.put("sequence", data.toLowerCase());
                } else {
                    seq.put("sequence", data.toUpperCase());
                }
            }
        }
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public void setFont (Font font) {
        super.setFont(font);
        setFixedCellHeight(getFontMetrics(font).getHeight());
        repaint();
    }

    /**
     * Adds an item to the GDEList's popup menu
     *
     * @param item the menu item to add.
     */
    public void addPopupMenuItem (JMenuItem item) {
        popup.add(item);
    }

    /**
     * Displays a window to edit the currently selected sequence's properties
     */
    public void getInfo() {
	Cell[] sequences = getData();
	if (sequences.length > 0) {
	    new GDESequenceWindow(datamodel, BLMain.getJFrame(), sequences);
	}
    }
    
    /**
     * Displays a window to edit the currently selected sequence's properties
     */
    public void delete() {
	final int[] deletionZone = getSelectedIndices();
	
	if ("true".equalsIgnoreCase(BLMain.getProperty("undo"))) {
	    UndoMulti undoResult = new UndoMulti();

	    for (int lineNum : deletionZone) {
		undoResult.addUndo(new UndoRowDeletion(datamodel, lineNum, datamodel.get(lineNum)));
	    }
	}
	datamodel.removeSequences(deletionZone);
    }
    
    /**
     * Returns the current list model.
     *
     * @return the current sequence list model.
     */
    public SequenceListModel getListModel () {
        return (SequenceListModel) super.getModel();
    }

    /**
     * Returns the currently selected data in the list.
     *
     * @return the current data for usage by commands
     */
    public Cell[] getData() {
        Cell seqAdd = null;                                     // the result of the current call to getSequence
	int count = 0;
	int countseq = 0;
	int[] indexlist = getSelectedIndices();
	Cell[] sequencelist = new Cell[indexlist.length];

        // itterate through the line numbers
        for (count = 0; count < indexlist.length; count++) {
            seqAdd = datamodel.get(indexlist[count]);
            
            // ensure that we do not add any null results
            if (seqAdd != null) {
                sequencelist[countseq] = seqAdd;
		countseq++;
            }
        }
	// ensure the proper array length
	if (countseq < indexlist.length) {
	    Cell[] temp = new Cell[countseq];
	    System.arraycopy(sequencelist, 0, temp, 0, countseq);
	    sequencelist = temp;
	}
	return sequencelist;
    }

    /**
     * Selects the group members of the currently selected sequence(s)
     */
    public void selectGroup() {
	int groupint[] = null;
	Cell current = null;
	List<Integer> group = null;

	for (int index : getSelectedIndices()) {
	    // get the currently selected sequence
	    current = datamodel.get(index);

	    // check if the currently selected sequence is a member of a sequence group
	    if (current.containsKey("group") && current.get("group") instanceof Integer && (group = datamodel.getgroup((Integer)current.get("group"))) != null) {
		// initialize the array to store the new selection indicies
		groupint = new int[group.size()];

		// itterate through the group and perform the mass selection
		for (int count = 0; count < group.size(); count++) {
		    groupint[count] = group.get(count);
		}

		// set the selection indicies
		setSelectedIndices(groupint);
	    }
	}
    }
}
