/*
 * SequenceListModel.java
 *
 * Created on June 11, 2009, 11:17 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.biolegato.gdesupport.canvas.list;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import javax.swing.ListModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.gdesupport.canvas.data.CellListener;
import org.biolegato.gdesupport.canvas.data.GDEModel;
import org.biolegato.gdesupport.canvas.data.GDEModelListener;

/**
 * A list model that is based on the sequence names in a sequence document.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class SequenceListModel implements ListModel, GDEModelListener {
    /**
     * The sequence document to obtain the sequence names from.
     */
    private GDEModel data;
    /**
     * The list of listeners to the data in the list.
     */
    private List<ListDataListener> listeners = new ArrayList<ListDataListener>();
    
    /**
     * Creates a new instance of SequenceListModel.
     *
     * @param data the sequence document to obtain the list data from.
     */
    public SequenceListModel(GDEModel data) {
        this.data = data;
        
        data.addListener(this);
    }

    /**
     * Returns the number of sequences in the sequence document.
     *
     * @return returns the number of sequences in the sequence document.
     */
    public int getSize() {
        return data.getLineCount();
    }

    /**
     * Returns the names of sequences based on their line number.
     *
     * @param index the line number to obtain the name of.
     * @return returns the name of sequence in the sequence document at the line number indicated by index.
     */
    public Object getElementAt(int index) {
        String name = "";
        Cell seq = data.get(index);
        
        if (seq != null) {
	    if (seq.get("group") != null && seq.get("group") instanceof Integer) {
		name += seq.get("group").toString() + " ";
	    } else {
		name += "_ ";
	    }
            name += seq.get("name").toString();
        }
        return name;
    }

    /**
     * Adds a data listener to the list model
     *
     * @param l the data listener to add.
     */
    public void addListDataListener(ListDataListener l) {
        listeners.add(l);
    }

    /**
     * Removes a data listener from the list model
     *
     * @param l the data listener to remove.
     */
    public void removeListDataListener(ListDataListener l) {
        listeners.remove(l);
    }

    /**
     * Passes any sequence modifications to the data listeners.
     *
     * @param source the source of the modification.
     * @param index the line number the modification occurred.
     * @param sequence the sequence that the modification occurred.
     * @param key the information within the sequence that has been changed.
     */
    public void sequenceChanged(GDEModel source, int index, Cell sequence, String key) {
        for (ListDataListener l : listeners) {
            l.contentsChanged(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, index, index));
        }
    }

    /**
     * Passes any sequence additions to the data listeners.
     *
     * @param source the source of the additon.
     * @param index the line number the additon occurred.
     * @param sequence the sequence added.
     */
    public void sequenceAdded(GDEModel source, int index, Cell sequence) {
        for (ListDataListener l : listeners) {
            l.intervalAdded(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, index, index));
        }
    }

    /**
     * Passes any sequence deletions to the data listeners.
     *
     * @param source the source of the deletion.
     * @param index the line number the deletion occurred.
     * @param sequence the sequence deleted.
     */
    public void sequenceRemoved(GDEModel source, int index, Cell sequence) {
        for (ListDataListener l : listeners) {
            l.intervalRemoved(new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, index, index));
        }
    }
    
}
