package org.biolegato.gdesupport.canvas.textarea;

/*
 * BLTextArea.java
 *
 * Created on August 18, 2008, 11:10 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.util.List;
import org.biolegato.gdesupport.canvas.GDECanvas;
import org.biolegato.gdesupport.canvas.data.GDEModel;
import org.biolegato.gdesupport.canvas.data.GDEModelListener;
import org.biolegato.gdesupport.canvas.colourmap.ColourMap;
import org.biolegato.gdesupport.canvas.colourmap.MonochromeColourMap;
import org.biolegato.gdesupport.canvas.listeners.CursorListener;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.LinkedList;
import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.core.main.BLMain;

/**
 * A general canvas with more functionality support than JTextArea.
 *
 * This canvas was originally created to support the rectangular selection model,
 * and has since supported rectangular selections, different colour schemes,
 * sequence documents, and many other features.
 * Please add more generic feature support as necessary.
 *
 * Also if you wish to add any functionality or handling specific to a plugin,
 * please feel free to subclass this textarea.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class BLTextArea extends BLComponent implements GDEModelListener, KeyListener, MouseMotionListener, MouseListener {

    /**
     * If true new text will overwrite existing text.
     */
    protected boolean insertMode = false;
    /**
     * Used to track the shift key.
     * If this is false, cancel the current selection on moving using the direction arrows.
     * If this is true, stretch the current selection on moving using the direction arrows.
     */
    protected boolean selectionMove = false;
    /**
     * Used to track mouse holds and releases.
     * If this is false, cancel the current selection on any mouse click.
     * If this is true, stretch the current selection on mouse movement.
     */
    protected boolean selectionMouse = false;
    /**
     * The current row of the caret
     */
    protected int row = 0;
    /**
     * The current column of the caret.
     */
    protected int column = 0;
    /**
     * The start column of the selection (no selection if sx = column)
     */
    private int sx = -1;
    /**
     * The start row of the selection (no selection if sy = row)
     */
    private int sy = -1;
    /**
     * Handles printing text
     */
    protected ColourMap currentMap = new MonochromeColourMap(Color.BLACK, Color.WHITE);
    /**
     * The right click menu for the text area.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * Linked list to store all cursor listeners
     */
    protected List<CursorListener> cursorListeners = new LinkedList<CursorListener>();
    /**
     * Linked list to store all mode change listeners
     */
    protected List<ModeListener> modeListeners = new LinkedList<ModeListener>();
    /**
     * Stores the state of whether the canvas's selection is active
     */
    private boolean active = false;
    /**
     * Used for the relationship between the data model and the text area
     */
    protected GDEModel datamodel;
    /**
     * The colour of invalid areas in the textarea.
     */
    protected final Color invalidzone = Color.RED;
    /**
     * Self-reference for inner classes.
     */
    public final BLTextArea blTextAreaSelf = this;
    /**
     * This constant is used for serialization purposes.
     */
    public static final long serialVersionUID = 7526472295622777004L;

    /**
     * Creates a new instance of BLTextArea
     */
    public BLTextArea(GDEModel datamodel) {
        // import all applicable constructor parameters
        this.datamodel = datamodel;
        
        // add the listeners
        getDataModel().addListener(this);
        addKeyListener(this);
        addMouseListener(this);
        addMouseMotionListener(this);

        // create popup menu
        addPopupMenuItem(new JMenuItem(new AbstractAction("Cut") {

            private static final long serialVersionUID = 7526472295622777029L;

            public void actionPerformed(ActionEvent e) {
                blTextAreaSelf.cut();
            }
        }));
        addPopupMenuItem(new JMenuItem(new AbstractAction("Copy") {

            private static final long serialVersionUID = 7526472295622777030L;

            public void actionPerformed(ActionEvent e) {
                blTextAreaSelf.copy();
            }
        }));
        addPopupMenuItem(new JMenuItem(new AbstractAction("Paste") {

            private static final long serialVersionUID = 7526472295622777031L;

            public void actionPerformed(ActionEvent e) {
                blTextAreaSelf.paste();
            }
        }));

        // set display
        setFont(BLMain.DEFAULT_FONT);
        setForeground(Color.BLACK);
        setBackground(Color.WHITE);
        setFocusTraversalKeysEnabled(false);
        refreshSize();
        setDoubleBuffered(true);
        repaint();
    }

////////////////////////
//********************//
//* LISTENER METHODS *//
//********************//
////////////////////////
    /**
     * Adds a cursor listener to the BLTextArea.
     *
     * @param listener the cursor listener to add.
     */
    public void addCursorListener(CursorListener listener) {
        cursorListeners.add(listener);
    }

    /**
     * Adds a cursor listener to the BLTextArea.
     *
     * @param listener the cursor listener to add.
     */
    public void addModeListener(ModeListener listener) {
        modeListeners.add(listener);
    }

    /**
     * Returns the Dataset associated with the BLTextArea
     * 
     * 
     * 
     * @return the SDatasetobject.
     */
    public GDEModel getDataModel() {
        return datamodel;
    }

///////////////////////
//*******************//
//* DISPLAY METHODS *//
//*******************//
///////////////////////
    /**
     * Changes the current colour map for unselected text.
     *
     * @param newMap the colour map to use
     */
    public void setColourMap(ColourMap newMap) {
        currentMap = newMap;
        repaint();
    }

    /**
     * Used to obtain all of the text selected within the document.
     *
     * @return the text currently selected in the document.
     */
    public Cell[] getData() {
        // TODO: getSelectedText
        int lineLength;
        String currentSequence = "";
        Cell current = null;
        int countseq = 0;
        int length = getMaxSY() - getMinSY() + 1;
        Cell[] result = null;

        if (!isSelectionEmpty()) {
            result = new Cell[length];
            for (int count = getMinSY(); count <= getMaxSY(); count++) {
                lineLength = getDataModel().getLineLength(count);
                if (getMinSX() < lineLength) {
                    current = getDataModel().get(count);
                    if (current != null) {
                        current = (Cell) current.clone();
                        currentSequence = (String) current.get("sequence");
                        if (getMaxSX() < lineLength) {
                            currentSequence = currentSequence.substring(0, getMaxSX());
                        }
                        if (getMinSX() > 0) {
                            currentSequence = currentSequence.substring(getMinSX());
                        }
                        if (!currentSequence.equals((String) current.get("sequence"))) {
                            current.put("sequence", currentSequence);
                        }
                        result[countseq] = current;
                        countseq++;
                    }
                }
            }
            // ensure the proper array length
            if (countseq < length) {
                Cell[] temp = new Cell[countseq];
                System.arraycopy(result, 0, temp, 0, countseq);
                result = temp;
            }
        } else {
            result = new Cell[0];
        }
        return result;
    }

    /**
     * Paints the current textarea (this uses the current clip bounds to determine the area to paint.
     *
     * @param gfx the graphics instance to paint the window to.
     */
    @Override
    public void paintComponent(Graphics gfx) {
        try {
            final Rectangle area = gfx.getClipBounds();   // get the area to paint
            final int startcol = Math.max(0, X2Column(area.x) - 1);
            final int startrow = Math.max(0, Y2Row(area.y) - 1);
            final int endcol = startcol + X2Column(area.width) + 2;
            final int endrow = Math.min(getDataModel().getLineCount(), startrow + Y2Row(area.height) + 2);
            final int xstart = column2X(startcol);
            final int ystart = row2Y(startcol);
            final int xend = column2X(endcol + 1);
            final int yend = row2Y(startcol + 1);
            // get the selection start position
            final int startSelect = Math.max(startcol, Math.min(sx, column));
            final int endSelect = Math.max(startcol, Math.max(sx, column));
            final int startSelectX = xstart + Math.round(columnSize() * Math.max(0, startSelect - startcol));
            final int endSelectX = xstart + Math.round(columnSize() * Math.max(0, endSelect - startcol));
            char[] print = null;
            Cell currentLine;
            ColourMap useMap = currentMap;

            int currenty;

            if (gfx != null) {
                // print the background
                gfx.setColor(getBackground());
                gfx.fillRect(xstart, ystart, xend - xstart, yend - ystart);

                // print each sequence
                for (int count = startrow; count < endrow; count++) {
                    currenty = row2Y(count);

                    // get the data to print
                    currentLine = getDataModel().get(count);
                    print = ((String) currentLine.get("sequence")).toCharArray();

                    // set the current colour map to the default current colour map
                    useMap = currentMap;

                    // use the sequence specific colour masks if available
                    if (currentLine.containsKey("mask") && currentLine.get("mask") instanceof ColourMap) {
                        useMap = (ColourMap) currentLine.get("mask");
                    }

                    if (isSelectedLine(count) && startcol < endSelect && endSelect >= 0 && startSelect >= 0 && endSelect - startSelect > 0) {
                        // draw the sequences
                        useMap.regularDrawString(this, gfx, xstart, currenty, currentLine, print, startcol, startSelect);
                        useMap.selectDrawString(this, gfx, startSelectX, currenty, currentLine, print, startSelect, endSelect);
                        useMap.regularDrawString(this, gfx, endSelectX, currenty, currentLine, print, endSelect, endcol);
                    } else {
                        useMap.regularDrawString(this, gfx, xstart, currenty, currentLine, print, startcol, endcol);
                    }
                }

                // draw the cursor.
                if (hasFocus()) {
                    gfx.setColor(getForeground());
                    gfx.drawLine(column2X(column), row2Y(row), column2X(column), row2Y(row + 1));
                }
            }
        } catch (Throwable th) {
            // don't halt printing other objects on crash
            th.printStackTrace();
        }
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public void setFont(Font font) {
        super.setFont(font);
        refreshSize();
        repaint();
    }

////////////////////////
//********************//
//* KEYBOARD METHODS *//
//********************//
////////////////////////
    /**
     * Processes the typing of keys within the text area
     *
     * @param event the KeyEvent for the key typed
     */
    public void keyTyped(KeyEvent event) {
        boolean canInsert = true;

        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                case KeyEvent.VK_DELETE:
                    break;
                case KeyEvent.VK_ENTER:
                    // now only changes the row (same effect as downkey, except clears selection)
                    if (row + 1 < getDataModel().getLineCount()) {
                        changePosition(false, Math.min(getDataModel().getLineLength(row + 1), column), row + 1);
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    break;
                default:
                    if (!isSelectionEmpty()) {
                        deleteSelection();
                    } else if (insertMode && getDataModel().getLineCount() > 0
                            && getDataModel().getLineLength(0) > 0) {
                        canInsert = delete(column, row, 1, 0);
                    }
                    if (isSelectionEmpty() || !insertMode || canInsert) {
                        insert(column, row, String.valueOf(event.getKeyChar()));
                        changePosition(false, column + 1, row);
                    }
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
        event.consume();
    }

    /**
     * Processes key presses within the text area
     *
     * @param event the KeyEvent for the key pressed
     */
    public void keyPressed(KeyEvent event) {
        int backspacecol = 0;

        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                    if (!isSelectionEmpty()) {
                        deleteSelection();
                    } else if (row > 0 || column > 0) {
                        if (column <= 0 && row > 0) {
                            backspacecol = getDataModel().getLineLength(row - 1);
                        }
                        if (column - 1 >= 0) {
                            if (delete(column - 1, row, 1, 0)) {
                                if (column <= 0 && row > 0) {
                                    changePosition(false, backspacecol, row - 1);
                                } else {
                                    changePosition(false, column - 1, row);
                                }
                            }
                        }
                    }
                    break;
                case KeyEvent.VK_DELETE:
                    if (!isSelectionEmpty()) {
                        deleteSelection();
                    } else if (getDataModel().getLineCount() > 0
                            && getDataModel().getLineLength(0) > 0) {
                        delete(column, row, 1, 0);
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            selectionMove = true;
                            break;
                        case KeyEvent.VK_LEFT:
                            if (column > 0) {
                                changePosition(selectionMove, column - 1, row);
                            } else if (row > 0) {
                                changePosition(selectionMove, getDataModel().getLineLength(row), row - 1);
                            }
                            break;
                        case KeyEvent.VK_RIGHT:
                            if (column + 1 <= getDataModel().getLineLength(row)) {
                                changePosition(selectionMove, column + 1, row);
                            } else if (row + 1 < getDataModel().getLineCount()) {
                                changePosition(selectionMove, 0, row + 1);
                            }
                            break;
                        case KeyEvent.VK_UP:
                            if (row > 0) {
                                changePosition(selectionMove, Math.min(getDataModel().getLineLength(row - 1), column), row - 1);
                            }
                            break;
                        case KeyEvent.VK_DOWN:
                            if (row + 1 < getDataModel().getLineCount()) {
                                changePosition(selectionMove, Math.min(getDataModel().getLineLength(row + 1), column), row + 1);
                            }
                            break;
                        case KeyEvent.VK_HOME:
                            changePosition(selectionMove, 0, row);
                            break;
                        case KeyEvent.VK_END:
                            changePosition(selectionMove, getDataModel().getLineLength(row), row);
                            break;
                        case KeyEvent.VK_COPY:
                            copy();
                            break;
                        case KeyEvent.VK_CUT:
                            cut();
                            break;
                        case KeyEvent.VK_PASTE:
                            paste();
                            break;
                        case KeyEvent.VK_INSERT:
                            insertMode = !insertMode;
                            for (ModeListener listener : modeListeners) {
                                listener.insertionMode(insertMode);
                            }
                            break;
                        default:
                            if (BLMain.debug) {
                                BLMain.warning("Unhandled key pressed --- getKeyChar() = " + ((int) event.getKeyChar()) + "\tgetKeyCode() = " + event.getKeyCode(), "BLTextArea");
                            }
                            break;
                    }
                    if (column >= getDataModel().getLineLength(row)) {
                        changePosition(selectionMove, getDataModel().getLineLength(row), row);
                    }
                    break;
                default:
                    deleteSelection();
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
        event.consume();
    }

    /**
     * Processes key releases within the text area
     *
     * @param event the KeyEvent for the key released
     */
    public void keyReleased(KeyEvent event) {
        try {
            switch (event.getKeyChar()) {
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            selectionMove = false;
                            break;
                        default:
                            break;
                    }
                    break;
                default:
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }
        event.consume();
    }

/////////////////////
//*****************//
//* MOUSE METHODS *//
//*****************//
/////////////////////
    /**
     * Handles mouse clicks.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set, in which case it stretches the selection).
     *
     * @param event the MouseEvent object corresponding to the click.
     */
    public void mouseClicked(MouseEvent event) {
        if (!event.isPopupTrigger()) {
            // calculate the row and column numbers from the X and Y co-ordinates.
            changePosition(selectionMove, X2Column(event.getX()), Y2Row(event.getY()));

            // request focus within the current window
            requestFocus();
            requestFocusInWindow();
        }
    }

    /**
     * Handles mouse button presses.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set).
     *
     * @param event the MouseEvent object corresponding to the press.
     */
    public void mousePressed(MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(this, event.getX(), event.getY());
        }
    }

    /**
     * Handles mouse button releases.
     *
     * @param event the MouseEvent object corresponding to the release.
     */
    public void mouseReleased(MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX(), event.getY());
        } else {
            selectionMouse = false;
        }
    }

    /**
     * Handles the mouse entering the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the enter.
     */
    public void mouseEntered(MouseEvent event) {
    }

    /**
     * Handles the mouse exiting the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the exit.
     */
    public void mouseExited(MouseEvent event) {
    }

    /**
     * Handles mouse drags.
     * Updates the selection and caret on drags.
     *
     * @param event the MouseEvent object corresponding to the drag.
     */
    public void mouseDragged(MouseEvent event) {
        if (!event.isPopupTrigger() || !isSelectionEmpty()) {
            // calculate the row and column numbers from the X and Y co-ordinates and update the position
            changePosition(selectionMouse || selectionMove, X2Column(event.getX()), Y2Row(event.getY()));

            selectionMouse = true;
            requestFocus();
            requestFocusInWindow();
        }
    }

    /**
     * Handles mouse movements.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the movement.
     */
    public void mouseMoved(MouseEvent event) {
    }

    /**
     * Refreshes the size of the textarea (for scroll size purposes)
     */
    protected void refreshSize() {
        final int width = (getDataModel().getLongestLine() + 1) * columnSize();
        final int height = rowSize() * getDataModel().getLineCount();

        Dimension size = new Dimension(width, height);
        setSize(size);
        setPreferredSize(size);
        revalidate();
    }

/////////////////////////////
//*************************//
//* CUSTOMIZATION METHODS *//
//*************************//
/////////////////////////////
    /**
     * Adds an item to the BLTextArea's right-click popup menu.
     *
     * @param act the menu item to add.
     */
    public void addPopupMenuItem(JMenuItem act) {
        popup.add(act);
    }

    /**
     * Removes an item from the BLTextArea's right-click popup menu.
     *
     * @param act the menu item to remove.
     */
    public void removePopupMenuItem(JMenuItem act) {
        popup.remove(act);
    }

/////////////////////////////////////
//*********************************//
//* EXTERNAL MANIPULATION METHODS *//
//*********************************//
/////////////////////////////////////
    /**
     * Copies content from the current Editable object.
     */
    public void copy() {
        if (!isSelectionEmpty()) {
            GDECanvas.setClipboard(getData());
        }
    }

    /**
     * Cuts content from the current Editable object.
     */
    public void cut() {
        if (!isSelectionEmpty()) {
            copy();
            deleteSelection();
        }
    }

    /**
     * Pastes content into the current Editable object.
     */
    public void paste() {
        deleteSelection();
        insert(column, row, GDECanvas.getClipboard());
    }

/////////////////////////////////////
//*********************************//
//* INTERNAL MANIPULATION METHODS *//
//*********************************//
/////////////////////////////////////
    /**
     * Inserts a string into the textarea's underlying Dataset
     * NOTE: this is a wrapper method for <b>insert</b>(<i>x</i>, <i>y</i>, <i>data</i>)
     * 
     * 
     * 
     * @param x the X co-ordinate (column number) to insert the string at.
     * @param y the Y co-ordinate (row number) to insert the string at.
     * @param data the string to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert(int col, int y, String data) {
        return getDataModel().insert(col, y, data);
    }

    /**
     * Inserts an array of sequences into the textarea's underlying Dataset
     * 
     * 
     * 
     * @param x the X co-ordinate (column number) to insert the sequences at.
     * @param y the Y co-ordinate (row number) to insert the sequences at.
     * @param data the array of sequences to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert(int col, int y, Cell[] data) {
        boolean result = true;

        for (int count = 0; count < data.length; count++) {
            if (y + count < getDataModel().getLineCount()) {
                result &= insert(col, y + count, (String) data[count].get("sequence"));
            } else {
                result &= getDataModel().addSequence(y + count, data[count]);
            }
        }
        return result;
    }

    /**
     * Deletes characters from the textarea's underlying Dataset
     * NOTE: deletions are performed in a sequential manner
     * 
     * 
     * 
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return a boolean corresponding to the result of the deletion (true = successful deletion)
     */
    public boolean delete(final int x, final int y, final int w, final int h) {
        return getDataModel().delete(x, y, w, h);
    }

    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     *
     * @return whether or not the deletion was performed.
     */
    public boolean deleteSelection() {
        final int x = getMinSX();
        final int y = getMinSY();
        final int w = getMaxSX() - x;
        final int h = getMaxSY() - y;
        boolean result = false;

        if (!isSelectionEmpty()) {
            if (delete(x, y, w, h)) {
                changePosition(false, x, y);
                result = true;
            }
        }
        return result;
    }

//////////////////////
//******************//
//* STATUS METHODS *//
//******************//
//////////////////////
    /**
     * Returns the insertion mode status of the textarea.
     *
     * Insertion mode is invoked by hitting the "INSERT" key.
     * While in insertion mode, if you type a character key in front of some text
     * the character key inserted will replace the first character of the text in front
     * of the insertion point.
     *
     * @return whether or not insertion mode is on.
     */
    public boolean getInsertMode() {
        return insertMode;
    }

////////////////////////
//********************//
//* LISTENER METHODS *//
//********************//
////////////////////////
    /**
     * Called when a sequence is added to a Dataset.
     * 
     * 
     * 
     * @param index the location the sequence was added.
     * @param sequence the sequence added.
     */
    public void sequenceAdded(GDEModel source, final int index, Cell sequence) {
        final int x = 0;				// the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();	// the width of the area to repaint
        final int y = row2Y(index);			// the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight() - y;  // the height of the area to repaint

        refreshSize();					// update the size of the textarea
        repaint(x, y, w, h);				// repaint the modified area of the textarea
    }

    /**
     * Called when a field in a sequence is modified.
     *
     * @param index the location of the sequence.
     * @param sequence the sequence modified.
     * @param key the value modified.
     */
    public void sequenceChanged(GDEModel source, int index, Cell sequence, String key) {
        final int x = 0;                            // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();   // the width of the area to repaint
        final int y = row2Y(index);                 // the Y co-ordinate to start repainting at
        final int h = rowSize();                    // the height of the area to repaint

        refreshSize();                              // update the size of the textarea
        repaint(x, y, w, h);                        // repaint the modified area of the textarea
    }

    /**
     * Called when a sequence is removed from a Dataset.
     * 
     * 
     * 
     * @param index the location (line number) where the sequence was removed from.
     * @param sequence the sequence removed from the SeDataset
     */
    public void sequenceRemoved(GDEModel source, int index, Cell sequence) {
        final int x = 0;				// the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();	// the width of the area to repaint
        final int y = row2Y(index);			// the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight() - y;  // the height of the area to repaint

        refreshSize();					// update the size of the textarea
        repaint(x, y, w, h);				// repaint the modified area of the textarea
    }

/////////////////////////
//*********************//
//* SELECTION METHODS *//
//*********************//
/////////////////////////
    /**
     * Updates/moves the cursor to the new position.
     * 
     * @param select whether or not the position should maintain selection status (i.e. true for SHIFT key).
     * @param newx the column co-ordinate of the new position.
     * @param newy the row co-ordinate of the new position.
     */
    protected void changePosition(boolean select, int newx, int newy) {
        final int maxlines = getDataModel().getLineCount() - 1;

        // ensure that the new row and column are valid
        newy = Math.max(0, Math.min(newy, maxlines));
        newx = Math.max(0, Math.min(newx, (getDataModel().getLineLength(newy))));

        final int startc = Math.min(newx, Math.min(column, sx));
        final int startr = Math.min(newy, Math.min(row, sy));
        final int endc = Math.max(newx, Math.max(column, sx)) + 1;
        final int endr = Math.max(newy, Math.max(row, sy)) + 1;

        final int x = column2X(startc);	// the X co-ordinate to start repainting at
        final int y = row2Y(startr);	// the Y co-ordinate to start repainting at
        final int w = column2X(endc);	// the width of the area to repaint
        final int h = row2Y(endr);	// the height of the area to repaint

        // update the row and column
        this.row = newy;
        this.column = newx;

        if (!select) {
            sx = column;
            sy = row;
        }

        // repaint the affected canvas area
        repaint(x, y, w, h);				// repaint the modified area of the textarea

        // update any cursor listeners
        for (CursorListener listener : cursorListeners) {
            listener.cursorChange(this, column, row);
        }

        scrollRectToVisible(new Rectangle(column2X(column), row2Y(row), 1, 1));
    }

    /**
     * Clears the current text selection
     */
    public void clearSelection() {
        changePosition(false, column, row);
    }

    /**
     * Tests if the selection shape is empty
     *
     * @return true if the selection is empty
     */
    public boolean isSelectionEmpty() {
        return (sx == column || sx < 0);
    }

    /**
     * Gets the minimum X co-ordinate of the selection area.
     *
     * @return Math.min(sx1, sx2)
     */
    protected int getMinSX() {
        return Math.min(sx, column);
    }

    /**
     * Gets the maximum X co-ordinate of the selection area.
     *
     * @return Math.max(sx1, sx2)
     */
    protected int getMaxSX() {
        return Math.max(sx, column);
    }

    /**
     * Gets the minimum Y co-ordinate of the selection area.
     *
     * @return Math.min(sy1, sy2)
     */
    protected int getMinSY() {
        return Math.min(sy, row);
    }

    /**
     * Gets the maximum Y co-ordinate of the selection area.
     *
     * @return Math.max(y1, y2)
     */
    protected int getMaxSY() {
        return Math.max(sy, row);
    }

    /**
     * This function is used to test if a line is selected.
     * Currently in this class, the test is just if the selection is within the bounds of sy1 and sy2.
     *
     * @param y the Y co-ordinate (or line number) to test.
     * @return true if the line is within the selection area, and should be printed as such.
     */
    protected boolean isSelectedLine(int y) {
        return (y >= getMinSY() && y <= getMaxSY());
    }
}
