/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas.textarea;

import java.awt.Color;
import java.awt.event.KeyEvent;
import java.util.Hashtable;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.JMenuItem;
import org.biolegato.gdesupport.canvas.data.Cell;
import java.util.regex.Pattern;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.GDECanvas;
import org.biolegato.gdesupport.canvas.GDECanvasObject;
import org.biolegato.gdesupport.canvas.data.GDEModel;
import org.biolegato.gdesupport.canvas.colourmask.ImportFileAction;
import org.biolegato.gdesupport.canvas.listeners.ModeListener;

/**
 * Custom text area for allowing box selection
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDETextArea extends BLTextArea implements GDECanvasObject {

    /**
     * Used for serialization
     */
    private static final long serialVersionUID = 7526472295622777017L;
    /**
     * The canvas associtated with the textarea.
     */
    private GDECanvas canvas = null;

    /**
     * Constructs a new instance of GDETextArea.
     */
    public GDETextArea (GDECanvas canvas) {
        super(canvas.getDataModel());
        
        // link the text area to its canvas
        this.canvas = canvas;
        
        // set up textarea colours
        setColourMap(ImportFileAction.getDefaultMap());
	// TODO fix the above line
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase () {
        String data = null;
        Cell[] seqList = null;

        // copy - paste section
        if ( ! isSelectionEmpty()) {
            seqList = getData();
            for (Cell seq : seqList) {
                data = seq.get("sequence").toString();

                if (data.length() >= 0) {
                    if (Character.isUpperCase(data.charAt(0))) {
                        seq.put("sequence", data.toLowerCase());
                    } else {
                        seq.put("sequence", data.toUpperCase());
                    }
                }
            }
            deleteSelection();
            insert(column, row, seqList);
        }
    }
    
    /**
     * Cuts content from the current Editable object.
     */
    @Override
    public void cut () {
        if ( ! isSelectionEmpty()) {
            copy();
            pdeleteSelection();
        }
    }

    /**
     * Pastes content into the current Editable object.
     */
    @Override
    public void paste () {
        pdeleteSelection();
        insert(column, row, GDECanvas.getClipboard());
    }

    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     * NOTE: ng stands for non-grouped
     *
     * @return whether or not the deletion was performed.
     */
    public boolean pdeleteSelection () {
	final int x = getMinSX();
	final int y = getMinSY();
	final int w = getMaxSX() - x;
	final int h = getMaxSY() - y;
	boolean result = false;
	
        if ( ! isSelectionEmpty() && pdelete(x, y, w, h)) {
	    changePosition(false, x, y);
	    result = true;
        }
        return result;
    }
    
    /**
     * This function is used to test if a line is selected.
     *
     * @param y the Y co-ordinate (or line number) to test.
     * @return true if the line is within the selection area, and should be printed as such.
     */
    public boolean isSelectedLine(int y) {
	boolean result = super.isSelectedLine(y);
	Cell current = null;
	List<Integer> group = null;
	
	if (!result) {
	    current = getDataModel().get(y);
	    if (current.get("group") instanceof Integer && (group = datamodel.getgroup((Integer)current.get("group"))) != null) {
		// itterate through the group and perform the mass insertion
		for (int gln : group) {
		    if (super.isSelectedLine(gln)) {
			result = true;
			break;
		    }
		}
	    }
	}
	return result;
    }

    /**
     * Checks a string against all of a sequence's protection settings.
     *
     * This is done by obtaining the sequence's protection settings,
     * the type of the sequence, and whether the text contains a character in any
     * protected character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean isProtectionsOn(Cell seq, String text) {
	return getProtectAlignment(seq, text) ||
	   getProtectAmbiguous(seq, text) ||
	   getProtectUnambiguous(seq, text);
    }
    
    /**
     * Checks a string against a sequence's alignment gap protection settings.
     *
     * This is done by obtaining the sequence's alignment gap protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * alignment gap character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectAlignment (Cell seq, String text) {
        boolean protect = false;
	char[] test;

        if (Boolean.TRUE.equals(seq.get("protect_align"))) {
            if (Cell.Type.DNA.equals(seq.get("type")) ||
		    Cell.Type.RNA.equals(seq.get("type"))) {
		
		test = text.toLowerCase().toCharArray();
		
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || (test[count] != 'a' && test[count] != 'c' && test[count] != 't'
			    && test[count] != 'g' && test[count] != 'u' && test[count] != 'r'
			    && test[count] != 'y' && test[count] != 'w' && test[count] != 's'
			    && test[count] != 'm' && test[count] != 'k' && test[count] != 'h'
			    && test[count] != 'b' && test[count] != 'v' && test[count] != 'i'
			    && test[count] != 'd' && test[count] != 'n');
		}
            } else if (Cell.Type.PROTEIN.equals(seq.get("type"))) {
		
		test = text.toLowerCase().toCharArray();
		
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == ' ' || test[count] == '\n'
			    || test[count] == '\t' || test[count] == '\r' || test[count] == '-';
		}
                protect = Pattern.matches("[\\s-]", text);
            }
        }
        return protect;
    }

    /**
     * Checks a string against a sequence's ambiguous character protection settings.
     *
     * This is done by obtaining the sequence's ambiguous character protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * ambiguous character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectAmbiguous (Cell seq, String text) {
        boolean protect = false;
	char[] test;

        if (Boolean.TRUE.equals(seq.get("protect_ambig"))) {
            if (Cell.Type.DNA.equals(seq.get("type")) ||
                Cell.Type.RNA.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == 'r' || test[count] == 'y' || test[count] == 'w'
			    || test[count] == 's' || test[count] == 'm' || test[count] == 'k'
			    || test[count] == 'h' || test[count] == 'b' || test[count] == 'v'
			    || test[count] == 'i' || test[count] == 'd' || test[count] == 'n';
		}
            } else if (Cell.Type.PROTEIN.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == 'b' || test[count] == '*'
			    || test[count] == 'z' || test[count] == 'x';
		}
            }
        }
        return protect;
    }

    /**
     * Checks a string against a sequence's unambiguous character protection settings.
     *
     * This is done by obtaining the sequence's unambiguous character protection settings,
     * the type of the sequence, and whether the text contains a character in the
     * unambiguous character class.
     *
     * @param seq the sequence to test against.
     * @param text the text to test.
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean getProtectUnambiguous (Cell seq, String text) {
        boolean protect = false;
	char[] test;

        if (Boolean.TRUE.equals(seq.get("protect_unambig"))) {
            if (Cell.Type.DNA.equals(seq.get("type")) ||
                Cell.Type.RNA.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || test[count] == 'a' || test[count] == 'c'
			    || test[count] == 't' || test[count] == 'g' || test[count] == 'u';
		}
            } else if (Cell.Type.PROTEIN.equals(seq.get("type"))) {
		test = text.toLowerCase().toCharArray();
		for (int count = 0; !protect && count < test.length; count++) {
		    protect = protect || (test[count] != 'b' && test[count] != '*'
			    && test[count] != 'z' && test[count] != 'x' && test[count] != ' '
			    && test[count] != '\n' && test[count] != '\t' && test[count] != '\r'
			    && test[count] != '-');
		}
            }
        }
        return protect;
    }
    
    /**
     * Inserts a string into the textarea's underlying SeqDoc
     * (NOTE: this should ONLY be called by sub-methods)
     *
     * @param x the X co-ordinate (column number) to insert the character at.
     * @param y the Y co-ordinate (row number) to insert the character at.
     * @param text the text to insert.
     * @return whether or not the insertion was successful.
     */
    public boolean insert (final int x, final int y, String text) {
        boolean result = true;
        Cell seq;
        Cell current = getDataModel().get(y);
	List<Integer> group;

	if (current.get("group") instanceof Integer && (group = datamodel.getgroup((Integer)current.get("group"))) != null) {
	    // itterate through the group and perform the mass insertion
	    for (int number : group) {
                if ( ! isProtectionsOn(datamodel.get(number), text)) {
                    result &= super.insert(x, number, text);
                }
	    }
	} else if ( ! isProtectionsOn(current, text)) {
	    result = super.insert(x, y, text);
	}
        return result;
    }

    /**
     * Removes text from a document.
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return whether the deletion was a success.
     */
    @Override
    public boolean delete (final int x, final int y, final int w, final int h) {
        boolean result = false;
        Cell current = null;
	List<Integer> group;

	for (int count = y; count <= y + h; count ++) {
            current = (Cell) getDataModel().get(count);
	    
	    if (current.get("group") instanceof Integer && (group = datamodel.getgroup((Integer)current.get("group"))) != null) {
		// itterate through the group and perform the mass insertion
		for (int gln : group) {
		    if ((gln < y || gln > y + h) && gln < getDataModel().getLineCount()) {
			result &= pdelete(x, gln, w, 0);
		    } else if (gln >= getDataModel().getLineCount()) {
			BLMain.error("Invalid row number: " + gln, "boolean GDETextArea.insert (char)");
		    }
		}
	    }
	    result = pdelete(x, count, w, 0);
	}
        return result;
    }
    
    /**
     * Removes text from one line of a document
     * (doesn't do any group processing - permission processing only).
     *
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @return whether the deletion was a success.
     */
    public boolean pdelete (final int x, final int y, final int w, final int h) {
	Cell current = null;
	String text = null;
	boolean result = false;

	for (int count = y; count <= y + h; count++) {
	    current = (Cell) getDataModel().get(count);
	    text = current.get("sequence").toString();
	    
	    if (x > 0 && x < text.length()) {
		text = text.substring(x);
	    } else if (x > text.length()) {
		text = "";
	    }
	    if (w > 0 && w < text.length()) {
		text = text.substring(0, w);
	    } else if (w < 0) {
		text = "";
	    }
	    
	    if ( ! isProtectionsOn(current, text)) {
		result = super.delete(x, count, w, 0);
	    }
	}
	
	return result;
    }
    
    public void linkCanvas(GDECanvas canvas) {
	this.canvas = canvas;
    }
    
    /**
     * Updates/moves the cursor to the new position.
     * 
     * @param select whether or not the position should maintain selection status (i.e. true for SHIFT key).
     * @param newx the column co-ordinate of the new position.
     * @param newy the row co-ordinate of the new position.
     */
    protected void changePosition (boolean select, int newx, int newy) {
	super.changePosition (select, newx, newy);
	
	if (select && canvas != null) {
	    canvas.selectionMade(this);
	}
    }
}
