/*
 * DataFormat.java
 *
 * Created on January 30, 2008, 11:54 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.formats;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.MappedByteBuffer;
import java.util.LinkedList;
import java.util.List;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.canvas.data.Cell;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

/**
 * Class to represent reading of data formats.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public abstract class DataFormat extends FileFilter {
    /**
     * Reference to genbank format for clipboard access
     */
    public static final DataFormat GENBANK = new GenBankFile2008();
    /**
     * The list of available file formats
     */
    private static final DataFormat[] formatList = new DataFormat[] {
        GENBANK, new GDEFile(), new GDEFlatfile(), new FastAFile()
    };
    /**
     * Creates a new instance of DataFormat
     */
    public DataFormat() {
    }
    
    /**
     * This function loads all of the available file formats into a JFileChooser.
     *
     * @param filechooser the JFileChooser to load the formats into.
     * @param defaultFormat the default file format for the file chooser.
     */
    public static void addFormats(JFileChooser filechooser, String defaultFormat) {
        DataFormat selected = null;	    // the defautl file format.
        
        // itterate through all file format subclasses
        for (DataFormat currentFormat : formatList) {
            // add the file filter to the JFileChooser
            filechooser.addChoosableFileFilter(currentFormat);

            // if the format is the default format specified in the method call,
            // set it to be the default selected format for the JFileChooser.
            if (currentFormat.getName().equals(defaultFormat)) {
                selected = currentFormat;
            }
        }
        
        // if the default file format was found, set the JFileChooser to have it selected.
        if (selected != null) {
            filechooser.setFileFilter(selected);
        }
    }
    
    /**
     * Reads a file's contained sequences using the file filter's translation.
     **
     * @param file the file to writer to.
     * @return the contents of the file
     * @throws IOException passes any exceptions from reading the file
     */
    public Cell[] readFile(File file) throws IOException {
        Cell[] result = new Cell[0];
        MappedByteBuffer stream = null;
        
        if (file != null && file.exists() && file.isFile() && file.length() > 0) {
                /*
                    possible faster method to read big files.
                    -----------------------------------------
                    stream = new FileInputStream(file.getPath()).getChannel().map(FileChannel.MapMode.READ_ONLY, 0, file.length());
                    result = translateFrom(new LineNumberReader(stream));
                 */
            result = translateFrom(new BufferedReader(new InputStreamReader(new BufferedInputStream(new FileInputStream(file)))));
        }
        return result;
    }
    
    /**
     * Writes a list of sequences to a file.
     *
     * @param file the file to writer to.
     * @param data the contents to writer.
     * @throws IOException throws any exceptions that occur while writing to the file
     */
    public void writeFile(File file, Cell[] rows) throws IOException {
        FileWriter writer = new FileWriter(file);   // the file writer object to write the file with.
        
        // translate and write the file.
        for (Cell line : rows) {
            writer.write(translateTo(line));
        }
        
        // flush and close the file writer buffer.
        writer.flush();
        writer.close();
    }
    
    /**
     * Obtains the name of the file format for File Chooser purposes
     *
     * @return the name of the file format
     */
    public String getName() {
        return "";
    }
    
    /**
     * Translates a string from the BioLegato internal format to the given file format.
     *
     * @param data the string to convert
     * @return the resulting string
     */
    public abstract String translateTo(Cell data);
    
    /**
     * Translates a string from the given file format into the BioLegato internal format.
     *
     * @param data the string to convert.
     * @return the resulting string.
     * @throws IOException any exeptions that occur while reading the stream are passed.
     */
    public abstract Cell[] translateFrom(java.io.BufferedReader data) throws IOException;
    
    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the string to test
     * @return whether or not the format is correct
     */
    public abstract boolean isFormat(java.io.Reader test);
    
    /**
     * Finds the filetype that corresponds to the given hashname (null if not successful).
     *
     * @param hashname the hashname to search for.
     * @return The result of the search (returns null on failure).
     */
    public static DataFormat getFormat(String hashname) {
        DataFormat found = null;	    // the data format found (null if not found).
        
        // ensure the name we are searching for is not null
        if (hashname != null) {
            // itterate through the list of plugins
            for (DataFormat currentFormat : formatList) {
                // break if successful match
                if (currentFormat != null && hashname.equalsIgnoreCase(currentFormat.getName())) {
                    found = currentFormat;
                    break;
                }
            }
        }
        return found;
    }
    
    /**
     * Used to auto-detect and parse files into BioLegato sequence objects.
     *
     * @param file the file to read.
     * @return The resulting sequences.
     */
    public static Cell[] auto(File file) {
        Cell[] result = new Cell[0];    // the resulting sequence set
        try {
            // autodetect the file.
            result = auto(new FileInputStream(file));
        } catch (IOException ioe) {
            // print an error message on failure
            BLMain.error("Bad file: " + file.getAbsolutePath(), "DataFormat.auto");
        }
        return result;
    }
    
    /**
     * Used to auto-detect and parse an input stream's data into BioLegato sequence objects.
     *
     * @param stream the stream to read.
     * @return The resulting sequences.
     */
    public static Cell[] auto(InputStream stream) {
        return auto(new BufferedReader(new InputStreamReader(new BufferedInputStream(stream))));
    }
    
    /**
     * Used to auto-detect and parse data into BioLegato sequence objects.
     *
     * @param data the string to convert
     * @return The resulting sequences.
     */
    public static Cell[] auto(java.io.BufferedReader data) {
        String line;
        List<Cell> makeSeq = null;	// used for if there is no acceptable format to translate the sequence string
        Cell[] result = null;		// the result of the file read.
        
        // ensure the data we are using is not null
        if (data != null) {
            // itterate through all file formats
            for (DataFormat currentFormat : formatList) {
                try {
                    // if the current format is a match, then translate using it.
                    if (currentFormat.isFormat(data)) {
                        result = currentFormat.translateFrom(data);
                        break;
                    }
                } catch (IOException e) {
                }
            }
        }
        
        if (result == null) {
            makeSeq = new LinkedList<Cell>();
            try {
                line = data.readLine();
                while (line != null) {
                    makeSeq.add(new Cell(line));
                    line = data.readLine();
                }
            } catch (IOException e) {
            }
            result = makeSeq.toArray(new Cell[0]);
        }
        
        // return the resulting sequences.
        return result;
    }
}
