package org.biolegato.gdesupport.formats;
/*
 * GDEFile.java
 *
 * Created on January 30, 2008, 11:58 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.util.List;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.gdesupport.canvas.data.Cell.Direction;
import org.biolegato.gdesupport.canvas.data.Cell.Strandedness;
import org.biolegato.gdesupport.canvas.data.Cell.Topology;
import org.biolegato.gdesupport.canvas.data.Cell.Type;
import org.biolegato.core.main.BLMain;
import org.biolegato.gdesupport.formats.DataFormat;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Hashtable;
import java.util.LinkedList;

/**
 * This class acts as a parser/translator for standard GDE files.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEFile extends DataFormat {

    /**
     * Creates a new instance of GDEFile
     */
    public GDEFile () {
    }

    /**
     * Translates a sequence into the GDE file format.
     *
     * @param seq the sequence to translate
     * @return the resulting string
     */
    public String translateTo (Cell seq) {
        StringBuffer result = new StringBuffer();

        // translate the data
        // NOTE: append is faster than + or concat operators
        if (seq != null) {
            result.append("{\nname            \"").append(
		    quote((String)seq.get("name"))).append(
			"\"\nlongname        \"").append(quote((String)seq.get("name"))).append(
			"\"\nsequence-ID     \"").append(quote((String)seq.get("accession"))).append(
			"\"\ncreation-date   \"").append((new SimpleDateFormat("MM/dd/yy kk:mm:ss")).format(new Date())).append(
			"\"\ndirection       ").append((Direction.FROM5TO3.equals(seq.get("direction")) ? "1" : "-1")).append(
			"\nstrandedness    ").append((Strandedness.SINGLE.equals(seq.get("strandedness")) ? "1" : "2")).append(
			"\ntype            ").append(blTypetoGDE((Type) seq.get("type"))).append(
			"\n").append((Topology.CIRCULAR.equals(seq.get("topology")) ? "circular        1\n" : "")).append(
			"offset          0\ngroupID         ").append((seq.get("group") != null && seq.get("group") instanceof Integer ? seq.get("group") : 0)).append(
			"\ncreator         \"").append(quote("")).append(
			"\"\ndescrip         \"").append(quote((String)seq.get("description"))).append(
			"\"\ncomments        \"").append(quote((String)seq.get("comments"))).append(
			"\"\nsequence        \"").append(quote((String)seq.get("sequence"))).append("\"\n}\n");
        }
        return result.toString();
    }

    /**
     * Translates data in the GDE file format to sequence objecte
     *
     * @param data the buffered reader to parse
     * @return the translated sequences
     * @throws IOException any exeptions that occur while reading the stream are passed
     */
    public Cell[] translateFrom (java.io.BufferedReader data) throws
            IOException {
	int groupID = 0;
        String line = "";
        String fname = "";
        String fvalue = "";
        StringBuffer fvalueBuffer;                                              // used for reading multiple lines
        List<Cell> result = new LinkedList<Cell>();
        Hashtable<String, Object> sequence = new Hashtable<String, Object>();

        // translate the data
        while ((line = data.readLine()) != null) {
            line = line.trim();
            if ( ! "".equals(line)) {
                if ((line.contains("}") || line.contains("{")) &&
                     ! sequence.isEmpty() && sequence.containsKey("sequence")) {
                    result.add(new Cell(sequence));
                    sequence.clear();
                    line = line.substring(line.indexOf('{') + 1).trim();
                }
                if (line.contains(" ") && ( ! line.contains("\t") ||
                                           line.indexOf(' ') <
                                           line.indexOf('\t'))) {
                    fname = line.substring(0, line.indexOf(' ')).trim();
                    fvalue = line.substring(line.indexOf(' ') + 1).trim();
                } else if (line.contains("\t")) {
                    fname = line.substring(0, line.indexOf('\t')).trim();
                    fvalue = line.substring(line.indexOf('\t') + 1).trim();
                } else {
                    fname = "";
                    fvalue = "";
                }
                if (fvalue.startsWith("\"")) {
                    fvalueBuffer = new StringBuffer();
                    fvalue = fvalue.substring(1);
                    fvalueBuffer.append(fvalue);
                    while (fvalue != null && ! fvalue.contains("\"")) {
                        // NOTE: append is faster than + or concat operators
                        fvalue = data.readLine();
                        fvalueBuffer.append(fvalue);
                    }
                    fvalue = fvalueBuffer.toString();
                    if (fvalue.contains("\"")) {
                        fvalue = fvalue.substring(0, fvalue.indexOf('\"'));
                    }
                }
                if ("".equals(fname) && "".equals(fvalue)) {
                } else if ("type".equals(fname)) {
                    if ("RNA".equalsIgnoreCase(fvalue)) {
                        sequence.put("type", Type.RNA);
                    } else if ("PROTEIN".equalsIgnoreCase(fvalue)) {
                        sequence.put("type", Type.PROTEIN);
                    } else if ("TEXT".equalsIgnoreCase(fvalue)) {
                        sequence.put("type", Type.TEXT);
                    } else if ("MASK".equalsIgnoreCase(fvalue)) {
                        sequence.put("type", Type.MASK);
                    } else {
                        sequence.put("type", Type.DNA);
                    }
                } else if ("groupID".equals(fname)) {
		    if (BLMain.testNumber(fvalue.trim().toCharArray())) {
			try {
			    groupID = Integer.parseInt(fvalue);
			    if (groupID > 0) {
				sequence.put("group", groupID);
			    }
			} catch (Exception ex) {
			}
		    }
                } else if ("circular".equals(fname)) {
                    if ("0".equals(fvalue)) {
                        sequence.put("topology", Topology.LINEAR);
                    } else {
                        sequence.put("topology", Topology.CIRCULAR);
                    }
                } else if ("direction".equals(fname)) {
                    if ("-1".equals(fvalue)) {
                        sequence.put("direction", Direction.FROM3TO5);
                    } else {
                        sequence.put("direction", Direction.FROM5TO3);
                    }
                } else if ("strandedness".equals(fname)) {
                    if ("1".equals(fvalue)) {
                        sequence.put("strandedness", Strandedness.SINGLE);
                    } else {
                        sequence.put("strandedness", Strandedness.DOUBLE);
                    }
                } else if ("creation-date".equals(fname)) {
                    sequence.put("creation-date", fvalue);
                } else if ("creator".equals(fname)) {
                    sequence.put("creator", fvalue);
                } else if ("sequence-ID".equals(fname)) {
                    sequence.put("accession", fvalue);
                } else if ("name".equals(fname)) {
                    sequence.put("name", fvalue);
                } else if ("longname".equals(fname)) {
                    sequence.put("longname", fvalue);
                } else if ("descrip".equals(fname)) {
                    sequence.put("description", fvalue);
                } else if ("comments".equals(fname)) {
                    sequence.put("comments", fvalue);
                } else if ("sequence".equals(fname)) {
                    sequence.put("sequence", fvalue);
                } else {
                    BLMain.warning("Unsupported attribute (" +
                                                           fname + " = " +
                                                           fvalue + " )",
                                                           "GDEFile plugin");
                }
            }
        }
        if ( ! sequence.isEmpty() && sequence.containsKey("sequence")) {
            result.add(new Cell(sequence));
        }
        return result.toArray(new Cell[0]);
    }

    /**
     * Determines whether or not a specified file is of type GDE file (based on extension)
     * Currently the only extension supported is ".gde".
     *
     * @param file the file to test
     * @return true if the file is of type GDE file (otherwise false)
     * @see javax.swing.filechooser.FileFilter#accept
     */
    public boolean accept (File file) {
        return (file.isDirectory() || file.getAbsolutePath().toLowerCase().
                endsWith(".gde"));
    }

    /**
     * Obtains the internal name of the file format
     * (this is great for hashtables storage and searching).
     *
     * @return the name of the file format
     */
    @Override
    public String getName () {
        return "gde";
    }

    /**
     * Returns a description of the file format that can be displayed to the user.
     *
     * @return the string description of the file format
     * @see javax.swing.filechooser.FileFilter#getDescription
     */
    public String getDescription () {
        return "GDE file (*.gde)";
    }

    /**
     * Quotes a GDE string (replaces " with ' and {} with [])
     * 
     * @param input	the string to quote
     * @return the quoted string
     */
    public static String quote (String input) {
        return input.replaceAll("\"", "\'").replaceAll("\\{", "\\[").replaceAll(
                "\\}", "\\]");
    }

    /**
     * Converts a sequence's type enum to the GDE file format
     *
     * @param type the type enum to convert
     * @return the GDE equivilent
     */
    private static String blTypetoGDE (Type type) {
        String result = "DNA";
        switch (type) {
            case RNA:
                result = "RNA";
                break;
            case PROTEIN:
                result = "PROTEIN";
                break;
            case MASK:
                result = "MASK";
                break;
            case TEXT:
                result = "TEXT";
                break;
        }
        return result;
    }

    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the reader to parse data from
     * @return whether or not the format is correct
     */
    @Override
    public boolean isFormat (Reader test) {
        int check = ' ';
        try {
            while (check == ' ' || check == '\t' || check == '\n' || check ==
                                                                     '\r') {
                test.mark(2);
                check = test.read();
            }
            test.reset();
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return (check == '{');
    }

}
