package org.biolegato.gdesupport.formats;
/*
 * GDEFlatfile.java
 *
 * Created on January 30, 2008, 11:58 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.util.List;
import org.biolegato.gdesupport.canvas.data.Cell;
import org.biolegato.gdesupport.canvas.data.Cell.Type;
import org.biolegato.gdesupport.formats.DataFormat;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.util.HashMap;
import java.util.LinkedList;

/**
 * This class acts as a parser/translator for GDE flat files.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEFlatfile extends DataFormat {

    /**
     * Creates a new instance of GDEFlatfile
     */
    public GDEFlatfile () {
    }

    /**
     * Translates a sequence into the GDE flat file format.
     *
     * @param seq the sequence to translate
     * @return the resulting string
     */
    public String translateTo(Cell seq) {
	char type = '#';
	StringBuffer result = new StringBuffer();
	java.util.Map<String, Object> sequence;
        // translate the data
	if (seq != null) {
            type = typeToFlatFile((Type) seq.get("type"));
            if (type != '%' && type != '@' && type != '\"') {
                type = '#';
            }
            // NOTE: append is faster than + or concat operators
            result.append(type).append(seq.get("name")).append("\n").append(seq.get("sequence")).append("\n");
	}
	return result.toString();
    }

    /**
     * Translates data in the GDE flat file format to sequence objecte
     *
     * @param data the buffered reader to parse
     * @return the translated sequences
     * @throws IOException any exeptions that occur while reading the stream are passed
     */
    public Cell[] translateFrom (java.io.BufferedReader data) throws
            IOException {
        String line = "";
        List<Cell> result = new LinkedList<Cell>();
        java.util.Map<String, Object> sequence = new HashMap<String, Object>();

        while ((line = data.readLine()) != null) {
            if (line.startsWith("#") || line.startsWith("%") || line.startsWith(
                    "@") || line.startsWith("\"")) {
                if ( ! sequence.isEmpty()) {
                    result.add(new Cell(sequence));
                    sequence.clear();
                }
                if (line.indexOf(' ') > 0) {
                    line = line.substring(0, line.indexOf(' '));
                }
                if (line.indexOf('\t') > 0) {
                    line = line.substring(0, line.indexOf('\t'));
                }
                if (line.indexOf('\n') > 0) {
                    line = line.substring(0, line.indexOf('\n'));
                }
                if (line.indexOf('\r') > 0) {
                    line = line.substring(0, line.indexOf('\r'));
                }
                sequence.put("type", flatFileToType(line.trim().charAt(0)));
                sequence.put("name", line.trim().substring(1));
            } else {
                sequence.put("sequence", (sequence.containsKey("sequence") ? sequence.get("sequence")
                                          : "") + line.trim());
            }
        }
        if ( ! sequence.isEmpty()) {
            result.add(new Cell(sequence));
        }
        return result.toArray(new Cell[0]);
    }

    /**
     * Determines whether or not a specified file is of type GDE flat file (based on extension).
     * Currently the only extension supported is ".flat".
     *
     * @param file the file to test
     * @return true if the file is of type GDE flat file (otherwise false)
     * @see javax.swing.filechooser.FileFilter#accept
     */
    public boolean accept (File file) {
        return (file.isDirectory() || file.getAbsolutePath().toLowerCase().
                endsWith(".flat"));
    }

    /**
     * Obtains the internal name of the file format
     * (this is great for hashtables storage and searching).
     *
     * @return the name of the file format
     */
    @Override
    public String getName () {
        return "flat";
    }

    /**
     * Returns a description of the file format that can be displayed to the user.
     *
     * @return the string description of the file format
     * @see javax.swing.filechooser.FileFilter#getDescription
     */
    public String getDescription () {
        return "GDE Flatfile (*.flat)";
    }

    /**
     * Converts a GDE flat file type character into a Cell.Type structure.
     * 
     * 
     * 
     * 
     * @param test the character to test for type
     * @return the resulting type
     */
    private static Type flatFileToType (char test) {
        Type result = Type.DNA;
        switch (test) {
            case '%':
                result = Type.PROTEIN;
                break;
            case '@':
                result = Type.MASK;
                break;
            case '\"':
                result = Type.TEXT;
                break;
        }
        return result;
    }

    /**
     * Converts a sequence's type enum to the GDE flat file format.
     *
     * @param type the type enum to convert
     * @return the GDE flat file equivilent
     */
    private static char typeToFlatFile (Type t) {
        char result = '#';
        switch (t) {
            case PROTEIN:
                result = '%';
                break;
            case MASK:
                result = '@';
                break;
            case TEXT:
                result = '\"';
                break;
        }
        return result;
    }
    
    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the reader to parse data from
     * @return whether or not the format is correct
     */
    @Override
    public boolean isFormat (Reader test) {
        int check = ' ';
        try {
            while (check == ' ' || check == '\t' || check == '\n' || check ==
                                                                     '\r') {
                test.mark(2);
                check = test.read();
            }
            test.reset();
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return (check == '#' || check == '%' || check == '@' || check == '\"');
    }

}
