/*
 * Main.java
 *
 * Created on November 7, 2007, 1:04 PM
 *
 * This is the file which contains all of the main classes, and functions for
 * running BioLegato.
 *
 * Current serializable number:
 * private static final long serialVersionUID = 7526472295622777040L;
 *
 * Released numbers:
 */
package org.biolegato.main;

import java.io.File;
import java.io.FileInputStream;
import java.io.OutputStreamWriter;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Pattern;
import javax.swing.AbstractAction;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import org.biolegato.menu.GDEMenuParser;

/**
 * The main program and generic function class.
 * <p>
 *  This class is used to generate the main window, do all startup processing,
 *  and run the program.  This class also contains most of the utility functions.
 * </p>
 **
 * @author Graham Alvare
 * @author Brian Fristensky
 * @version 0.7.5 25-Mar-2011
 */
public final class BLMain {

    public static enum OS {

        BSD,
        LINUX,
        HP_UX,
        MACOS,
        OSX,
        SOLARIS,
        WINDOWS_9X,
        WINDOWS_NT,
        UNIX;

        public boolean isWindows() {
            return (this == WINDOWS_9X || this == WINDOWS_NT);
        }

        public boolean isUNIX() {
            return (this != WINDOWS_9X && this != WINDOWS_NT);
        }

        /**
         * Detects the current operating system
         * Check out: http://lopica.sourceforge.net/os.html
         **
         * @return the current operating system
         */
        public static OS detectOS() {
            OS result = UNIX;
            String osName = System.getProperty("os.name").toLowerCase();

            if (osName.startsWith("windows")) {
                result = WINDOWS_NT;
                if (osName.startsWith("windows 9") || osName.equals("windows me")) {
                    result = WINDOWS_9X;
                }
            } else if (osName.startsWith("solaris") || osName.startsWith("sunos")) {
                result = SOLARIS;
            } else if (osName.startsWith("linux")) {
                result = LINUX;
            } else if (osName.endsWith("bsd")) {
                result = BSD;
            } else if (osName.startsWith("hp-ux")) {
                result = HP_UX;
            } else if (osName.startsWith("mac os")) {
                result = MACOS;
            } else if (osName.startsWith("mac os x")) {
                result = OSX;
            } else {
                BLMain.warning("Did not detect your OS!  Defaulting to UNIX: " + System.getProperty("os.name")
                        + "   version: " + System.getProperty("os.version"), "main");
            }
            return result;
        }
    }

    public static enum ARCH {

        X86,
        AMD64,
        ALPHA,
        ARM,
        MIPS,
        SPARC,
        PPC,
        PPC64,
        UNKNOWN;

        /**
         * Detects the current system architecture
         * Check out: http://lopica.sourceforge.net/os.html
         **
         * @return the current system architecture
         */
        public static ARCH detectARCH() {
            ARCH result = UNKNOWN;
            String osArch = System.getProperty("os.arch").toLowerCase();

            if (osArch.equals("sparc")) {
                result = SPARC;
            } else if (osArch.equals("arm")) {
                result = ARM;
            } else if (osArch.equals("alpha")) {
                result = ALPHA;
            } else if (osArch.equals("mips")) {
                result = MIPS;
            } else if (osArch.equals("amd64") || osArch.equals("x86_64")) {
                result = AMD64;
            } else if (osArch.equals("x86") || (osArch.startsWith("i") && osArch.endsWith("86"))) {
                result = X86;
            } else if (osArch.equals("ppc") || osArch.startsWith("power")) {
                if (!osArch.endsWith("64")) {
                    result = PPC;
                } else {
                    result = PPC64;
                }
            } else {
                BLMain.warning("Did not detect your system archetecture: " + System.getProperty("os.arch"), "main");
            }
            return result;
        }
    }
/////////////////////////
//*********************//
//* PROGRAM CONSTANTS *//
//*********************//
/////////////////////////
    /**
     * This constant stores an empty string (to avoid recreation)
     */
    public static final String EMPTY_STRING = "";
    /**
     * This constant stores the program's name
     */
    public static final String NAME = "BioLegato";
    /**
     * This constant is set to the path of BioLegato.jar
     * The value of this constant determined at runtime.
     */
    public static final String PROGRAM_DIR =
            new File(
            BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()).isDirectory()
            ? BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()
            : new File(
            new File(
            BLMain.class.getProtectionDomain().getCodeSource().getLocation().getPath()).getParent()).getPath();
    /**
     * This constant is set to the path of where biolegato was run
     * The value of this constant determined at runtime.
     */
    public static final String CURRENT_DIR = System.getProperty("user.dir");
    /**
     * This constant is used to keep track of version changes.
     */
    public static final String VERSION = "0.7.5";
    /**
     * Stores the user's home directory
     */
    public static final String HOME_DIR = System.getProperty("user.home");
    /**
     * The amount of time between clicks to be considered a double click.
     */
    public static final int DOUBLE_CLICK_TIME = 300;
    /**
     * Stores the current operating system
     */
    public static final OS CURRENT_OS = OS.detectOS();
    /**
     * Stores the current machine architecture
     */
    public static final ARCH CURRENT_ARCH = ARCH.detectARCH();
    /**
     * This constant is used for Serialization
     */
    public static final long serialVersionUID = 7526472295622776147L;
    /**
     * The menu item for BioLegato's "About..."
     */
    public static final JMenuItem ABOUT_MENUITEM = new JMenuItem(new AbstractAction("About...") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_A));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {
            JOptionPane.showMessageDialog(window,
                    "BioLegato version "
                    + VERSION
                    + "\nby Graham Alvare and Brian Fristensky\nUniveristy of Manitoba 2008-2010",
                    "About BioLegato",
                    JOptionPane.QUESTION_MESSAGE);
        }
    });
    /**
     * The menu item for BioLegato's "Exit..."
     */
    public static final JMenuItem EXIT_MENUITEM = new JMenuItem(new AbstractAction("Exit") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_X));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {
            window.dispose();
        }
    });
    /**
     * Stores the properties for BioLegato.
     * The current properties registered are:
     *
     * <table>
     *************************************************************************************************
     *	<tr><th>Property</th>               <th>Used internally</th>    <th>Description</th></tr>
     *************************************************************************************************
     *  <tr><td>debug</td>                  <td>Yes</td>                <td>determines whether or not to display debug information</td></tr>
     *	<tr><td>plugins</td>                <td>Yes</td>                <td>sets the plugin directory</td></tr>
     *	<tr><td>temp</td>                   <td>Yes</td>                <td>sets the temporary files directory</td></tr>
     *	<tr><td>font.size</td>              <td>Yes</td>                <td>determines the default font size for objects in BioLegato</td></tr>
     *	<tr><td>font.type</td>              <td>Yes</td>                <td>determines the default font type for objects in BioLegato</td></tr>
     *	<tr><td>shell.name</td>             <td>Yes</td>                <td>determines the shell to execute</td></tr>
     *	<tr><td>shell.parameter</td>        <td>Yes</td>                <td>determines the parameter for directing the shell to execute a command</td></tr>
     *	<tr><td>user.plugins</td>	    <td>Yes</td>                <td>determines whether or not to read the user's plugin files (located in the user's home directory</td></tr>
     *	<tr><td>canvas</td>                 <td>Yes</td>                <td>determines the canvas to show on startup (if multiple canvases available)</td></tr>
     *	<tr><td>GDE.menu</td>               <td>No</td>                 <td>determines whether or not to read the GDE menu files (backwards compatibility mode)</td></tr>
     *	<tr><td>GDE.help.viewer</td>        <td>No</td>                 <td>determines which file to feed the help file to</td></tr>
     *	<tr><td>GDE.help.path</td>          <td>No</td>                 <td>the location to search for the help files</td></tr>
     * </table>
     *
     * This properties class will read properties files from the directory containing BioLegato, the user directory and finally
     * the directory BioLegato was launched from.  Please note that you can disable reading of properties files
     * from the user directory via. "user.properties"; likewise you can disable reading properties files in the directory BioLegato
     * was launched from via. "pwd.properties"
     *
     * NOTE: for path properties BioLegato will replace all $'s with the appropriate environment variables if set.
     *
     * @author Graham Alvare
     * @author Brian Fristensky
     */
    public static final Properties DEFAULT_PROPERTIES = new Properties() {

        /**
         * Used for Serialization
         */
        private static final long serialVersionUID = 7526472295622777034L;

        {
            /* determines whether or not to display debug information */
            setProperty("debug", "false");
            /* sets the plugin directory */
            setProperty("plugins", BLMain.PROGRAM_DIR + File.separator + "plugins");
            /* sets the temporary file directory */
            setProperty("temp", BLMain.CURRENT_DIR);
            /* determines the default font size */
            setProperty("font.size", "12");
            /* determines the default font type */
            setProperty("font.type", "Monospace");
            /* determines the shell to execute */
            setProperty("shell.name", BLMain.CURRENT_OS == BLMain.OS.WINDOWS_9X ? "command.com"
                    : (BLMain.CURRENT_OS == BLMain.OS.WINDOWS_NT ? "cmd.exe" : "/bin/sh"));
            /* determines the parameter for directing the shell to execute a command */
            setProperty("shell.parameter", (BLMain.CURRENT_OS.isWindows() ? "/C" : "-c"));
            /* determines whether or not to read the user's plugin files (located in the user's home directory */
            setProperty("user.plugins", "true");
            /* determines the default canvas to show on startup */
            setProperty("canvas", "Table");
            /* determines the name of new sequences */
            setProperty("default.sequencename", "New sequence");
            //******************//
            //* GDE PROPERTIES *//
            //******************//
                /* determines whether or not to read the GDE menu files (backwards compatibility mode) */
            setProperty("GDE.menu", "true");
            /* determines which file to feed the help file to */
            setProperty("GDE.help.viewer", "$BIRCH/script/gde_help_viewer.csh");
            /* the location to search for the help files */
            setProperty("GDE.help.path", (System.getenv("GDE_HELP_DIR") != null ? System.getenv("GDE_HELP_DIR")
                    : ""));
            setProperty("seqcolourdir", BLMain.PROGRAM_DIR + File.separator + "colourmasks");
        }
    };
/////////////////
//*************//
//* VARIABLES *//
//*************//
/////////////////
    /**
     * Reference to the main program window
     */
    private static JFrame window = null;
    /**
     * Stores the properties for BioLegato.
     * The current properties registered are:
     *
     * <table>
     *	<tr><th>Property</th>               <th>Used internally</th>    <th>Description</th></tr>
     *      <tr><td>debug</td>                  <td>Yes</td>                <td>determines whether or not to display debug information</td></tr>
     *	<tr><td>plugins</td>                <td>Yes</td>                <td>sets the plugin directory</td></tr>
     *	<tr><td>temp</td>                   <td>Yes</td>                <td>sets the temporary files directory</td></tr>
     *	<tr><td>font.size</td>              <td>Yes</td>                <td>determines the default font size for objects in BioLegato</td></tr>
     *	<tr><td>font.type</td>              <td>Yes</td>                <td>determines the default font type for objects in BioLegato</td></tr>
     *	<tr><td>shell.name</td>             <td>Yes</td>                <td>determines the shell to execute</td></tr>
     *	<tr><td>shell.parameter</td>        <td>Yes</td>                <td>determines the parameter for directing the shell to execute a command</td></tr>
     *	<tr><td>user.plugins</td>	    <td>Yes</td>                <td>determines whether or not to read the user's plugin files (located in the user's home directory</td></tr>
     *	<tr><td>default.canvas</td>         <td>Yes</td>                <td>determines the default canvas to show on startup</td></tr>
     *	<tr><td>GDE.menu</td>               <td>No</td>                 <td>determines whether or not to read the GDE menu files (backwards compatibility mode)</td></tr>
     *	<tr><td>GDE.help.viewer</td>        <td>No</td>                 <td>determines which file to feed the help file to</td></tr>
     *	<tr><td>GDE.help.path</td>          <td>No</td>                 <td>the location to search for the help files</td></tr>
     * </table>
     *
     * This properties class will read properties files from the directory containing BioLegato, the user directory and finally
     * the directory BioLegato was launched from.  Please note that you can disable reading of properties files
     * from the user directory via. "user.properties"; likewise you can disable reading properties files in the directory BioLegato
     * was launched from via. "pwd.properties"
     *
     * NOTE: for path properties BioLegato will replace all $'s with the appropriate environment variables if set.
     **
     * @author Graham Alvare
     * @author Brian Fristensky
     */
    private static Properties properties = new Properties(DEFAULT_PROPERTIES) {

        /**
         * Changes a property within the BLProperties object
         **
         * @param key the key of the property to alter.
         * @param value the new value of the property.
         * @return the old value of the property changed (null if no previous value).
         */
        @Override
        public Object setProperty(String key, String value) {
            Object result = null;

            // ensure that the key and value parameters are not null
            if (key != null && value != null) {
                // make sure the key is always lower case
                key = key.toLowerCase();

                result = super.setProperty(key, value);  // the old value of the property changed
            }

            // return the function
            return result;
        }

        /**
         * Obtains a property from the BLProperties object.
         **
         * @param key the key of the property.
         * @return the current value of the property (returns "" if not set).
         */
        @Override
        public String getProperty(String key) {
            // make sure the key is always lower case
            key = key.toLowerCase();

            String result = super.getProperty(key);
            if (result == null) {
                result = "";
            }
            return result;
        }
    };
    /**
     * The tab pane storing all of the canvases.
     * This is stored so as to be able to get the current canvas for retrieving
     * the current data selection.
     */
    private static DataCanvas canvas = null;
    /**
     * Stores the main menu for the program
     */
    private static JMenuBar menu = null;
    /**
     * The current directory to use for open and save file choosers.
     * This variable will cache the last opened directory, so as to maintain consistency.
     */
    private static File currentPWD = new File(CURRENT_DIR);
    /**
     * Stores all of the menu headings in BioLegato's main window.
     * This hashtable is used to add menu items to BioLegato's menu headings.
     * The key of the hashtable corresponds to the label of the menu heading.
     * The value of the hashtable is the menu heading's object.
     */
    private static Map<String, JMenu> menuHeadings =
            new HashMap<String, JMenu>();
    /**
     * Stores all of the plugins loaded into BioLegato
     */
    private static Map<String, PluginWrapper> plugins = new HashMap<String, PluginWrapper>();
    /**
     * Stores information regarding the usage of the debug command
     * NOTE: This should always be accessed using Main.debug
     */
    public static boolean debug = false;

/////////////////////////
//*********************//
//* STARTUP FUNCTIONS *//
//*********************//
/////////////////////////
    /**
     * Starts up BioLegato.
     *
     * @param args the command line arguments for BioLegato.
     */
    public static void main(String[] args) {
//////////////////////////
//**********************//
//* FUNCTION VARIABLES *//
//**********************//
//////////////////////////
        // RESERVED (general purpose - inner scoped)
        // fileIn
        // currentCanvas
        String value = "";	    // the value of the current argument parameter
        String argument = null;	    //
        List<File> dataAdd = new LinkedList<File>();
        File propertiesFileTemp;

/////////////////////////////////
//*****************************//
//* PROPERTIES INITIALIZATION *//
//*****************************//
/////////////////////////////////
        // read properties files
        if (System.getenv("BL_PROPERTIES") != null) {
            propertiesFileTemp = new File(System.getenv("BL_PROPERTIES"));
        } else {
            propertiesFileTemp = new File(BLMain.PROGRAM_DIR + File.separator + ".blproperties");
        }
        if (propertiesFileTemp.exists()) {
            try {
                properties.load(new FileInputStream(propertiesFileTemp));
            } catch (Throwable e) {
                e.printStackTrace(System.err);
            }
        }

/////////////////////////
//*********************//
//* PROCESS ARGUMENTS *//
//*********************//
/////////////////////////

        // load the plugins and file formats
        PluginLoader.loadPlugins(plugins, envreplace(getProperty("plugins")));

        // ensure that args is not null
        if (args != null) {
            // itterate through the command arguments
            for (String rawarg : args) {
                // discard null arguments
                if (rawarg != null) {
                    File fileIn = null;	    //

                    // create a new file object to test if the argument is a file
                    fileIn = new File(rawarg);

                    // if the argument is a file name then read the file; otherwise,
                    // parse the argument
                    if (fileIn != null && fileIn.exists() && fileIn.isFile()
                            && fileIn.canRead()) {
                        dataAdd.add(fileIn);
                    } else if (rawarg.startsWith("/") || rawarg.startsWith(
                            "-")) {
                        // copy the command line argument
                        argument = rawarg;

                        // trim the argument
                        if (argument.startsWith("--") && argument.length() > 2) {
                            argument = argument.substring(2);
                        } else if ((argument.startsWith("/")
                                || argument.startsWith("-"))
                                && argument.length() > 1) {
                            argument = argument.substring(1);
                        }

                        // If there is an equals '=' sign then the argument has a value.  Set the value
                        // variable to the arguments' value for further parsing
                        if (argument.indexOf('=') > 0) {
                            value = argument.substring(argument.indexOf('=') + 1);
                            argument = argument.substring(0, argument.indexOf(
                                    '='));
                        }

                        // make the argument lower case for better parsing
                        argument = argument.toLowerCase().trim();

                        // process the argument
                        if ("help".equals(argument) || "h".equals(argument)
                                || "?".equals(argument)) {
                            ////////////
                            //********//
                            //* HELP *//
                            //********//
                            ////////////
                            // show BioLegato's usage
                            System.out.println(
                                    "Usage: biolegato [options] [files]\n"
                                    + "Use --optionlist  to see a detailed list of options\n"
                                    + "Use --manpage     to see BioLegato's manpage");
                            System.exit(0);
                        } else if ("optionlist".equals(argument)) {
                            ///////////////////
                            //***************//
                            //* OPTION LIST *//
                            //***************//
                            ///////////////////
                            // show BioLegato's list of options
                            System.out.println(
                                    "+-------------+------------------------------------------------------+\n"
                                    + "| Option      | Description                                          |\n"
                                    + "+-------------+------------------------------------------------------+\n"
                                    + "| help,?      | Displays usage information for BioLegato             |\n"
                                    + "| manpage,man | Displays the manpage entry for BioLegato             |\n"
                                    + "| optionlist  | Displays the list of options for using BioLegato     |\n"
                                    + "| version,v   | Displays the version information for BioLegato       |\n"
                                    + "| debug       | Enables BioLegato's debug mode                       |\n"
                                    + "| plugins     | Displays a list of loaded plugins                    |\n"
                                    + "| properties  | Displays a list of BioLegato's properties            |\n"
                                    + "+-------------+------------------------------------------------------+\n");
                            System.exit(0);
                        } else if ("manpage".equals(argument)
                                || "man".equals(argument)) {
                            ///////////////
                            //***********//
                            //* MANPAGE *//
                            //***********//
                            ///////////////
                            // show BioLegato's usage
                            System.out.print(
                                    "NAME\n"
                                    + "    Bio Legato - A customizable GUI for running programs\n"
                                    + "\n"
                                    + "VERSION\n"
                                    + "    Version ");
                            System.out.println(VERSION);
                            System.out.println("\n"
                                    + "SYNOPSIS\n"
                                    + "    biolegato [options] [files]\n"
                                    + "\n"
                                    + "DESCRIPTION\n"
                                    + "    BioLegato is a customizable GUI for running programs.\n"
                                    + "    Its initial intent is to be a replacement for GDE; however\n"
                                    + "    with its large plugin API, it may be customized to run more\n"
                                    + "    than just CLI programs.\n"
                                    + "\n"
                                    + "    With the versitility of BioLegato's plugin interface, it\n"
                                    + "    supports a wide range of file formats which may be added to\n"
                                    + "    at any time through addition of plugins.\n"
                                    + "\n"
                                    + "OPTIONS\n"
                                    + "  NOTE: All command line parameters are case insensitive\n"
                                    + "        and may optionally begin with -, --, or / to prevent\n"
                                    + "        confusion with filenames\n"
                                    + "\n"
                                    + "    help,h,?\n"
                                    + "        Displays usage information for BioLegato\n"
                                    + "    manpage,man\n"
                                    + "        Displays the manpage entry for BioLegato (this screen)\n"
                                    + "    optionlist\n"
                                    + "        Displays the list of options for using BioLegato\n"
                                    + "    version,v\n"
                                    + "        Displays the version information for BioLegato\n"
                                    + "    debug\n"
                                    + "        Enables BioLegato's debug mode\n"
                                    + "    plugins\n"
                                    + "        Displays a list of loaded plugins\n"
                                    + "    properties\n"
                                    + "        Displays a list of BioLegato's properties\n"
                                    + "\n"
                                    + "USAGE EXAMPLES\n"
                                    + "    biolegato\n"
                                    + "\n"
                                    + "    biolegato insequence.gb\n"
                                    + "\n"
                                    + "    biolegato --debug\n"
                                    + "    biolegato /debug\n"
                                    + "    biolegato -debug\n"
                                    + "\n"
                                    + "    biolegato --plugins --properties --debug\n"
                                    + "\n"
                                    + "    biolegato --plugins /properties -debug insequence.gb\n"
                                    + "\n"
                                    + "ENVIRONMENT VARIABLES\n"
                                    + "    BioLegato searches for the following environment variables.  If they don't exist BioLegato sets them to defaults.\n"
                                    + "\n"
                                    + "      BL_HOME\n"
                                    + "        Default: the path of the current working directory\n"
                                    + "        Description: the path to reference BioLegato from\n"
                                    + "\n"
                                    + "      BL_MASKS\n"
                                    + "        Default: uses .blproperties value\n"
                                    + "        Description: the directory to populate the default colourmask\n"
                                    + "                     list\n"
                                    + /*                                    "\n" +
                                    "      BL_MENUS\n" +
                                    "        Default: $BL_HOME/menus\n" +
                                    "        Description: the locations to load the BioLegato menus from\n" +*/ "\n"
                                    + "      BL_PROPERTIES\n"
                                    + "        Default: $BL_HOME/properties\n"
                                    + "        Description: the locations to load the BioLegato properties file from\n"
                                    + "\n"
                                    + "      GDE_HELP_DIR\n"
                                    + "        Default: null\n"
                                    + "        Description: specifies a path to search for the main .GDEmenus file\n"
                                    + "                     (NOTE: this is GDE_Menus.jar plugin specific)\n"
                                    + "\n"
                                    + "      GDE_MAKEMENUS_DIR\n"
                                    + "        Default: null\n"
                                    + "        Description: specifies a path to use as the root for parsing GDE\n"
                                    + "                     format menus in the same way as BIRCH's makemenus.py\n"
                                    + "\n"
                                    + "      BL_DEBUG\n"
                                    + "        Default: uses .blproperties value OR command switch --debug\n"
                                    + "        Description: set to \"true\" to enable debug mode\n"
                                    + "\n"
                                    + "      NOTE: in properties files, BL_DIR points to the directory where\n"
                                    + "            BioLegato is stored\n"
                                    + "\n"
                                    + "PROPERTIES\n"
                                    + "    BioLegato supports the followind properties:\n"
                                    + "\n"
                                    + "      Property             Description\n"
                                    + "        debug                determines whether or not to display debug information\n"
                                    + "        plugins              sets the plugin directory\n"
                                    + "        temp                 sets the temporary files directory\n"
                                    + "        font.size            determines the default font size for objects in BioLegato\n"
                                    + "        font.type            determines the default font type for objects in BioLegato\n"
                                    + "        shell.name           determines the shell to execute\n"
                                    + "        shell.parameter      determines the parameter for directing the shell to execute a command\n"
                                    + "        user.plugins         determines whether or not to read the user's plugin files (located in the user's home directory\n"
                                    + "        default.canvas       determines the default canvas to show on startup\n"
                                    + "\n"
                                    + "    This properties class will read properties files from the directory containing BioLegato, the user directory and finally\n"
                                    + "    the directory BioLegato was launched from.  Please note that you can disable reading of properties files\n"
                                    + "    from the user directory via. \"user.properties\"; likewise you can disable reading properties files in the directory BioLegato\n"
                                    + "    was launched from via. \"pwd.properties\""
                                    + "\n"
                                    + "    NOTE: for path properties BioLegato will replace all $'s with the appropriate environment variables if set.\n"
                                    + "\n"
                                    + "SUPPORTED PLUGINS\n"
                                    + "    Currently BioLegato supports the following main types of Plugins:\n"
                                    + "\n"
                                    + "        Canvases\n"
                                    + "           extends the class: org.biolegato.core.plugintypes.DataCanvas\n"
                                    + "        File formats\n"
                                    + "           extends the class: org.biolegato.core.plugintypes.DataFormat\n"
                                    + "        Menu types\n"
                                    + "           extends the class: org.biolegato.core.plugintypes.MenuType\n"
                                    + "\n"
                                    + "    For more information about plugins, please consult the BioLegato API\n"
                                    + "\n"
                                    + "FILES\n"
                                    + "    biolegato\n"
                                    + "        Script to run BioLegato\n"
                                    + "    biolegato.jar\n"
                                    + "        BioLegato's core java code\n"
                                    + "    changelog.txt\n"
                                    + "        BioLegato's revision history\n"
                                    + "    plugins/GDE.jar"
                                    + "        Provides a GDE style multiple alignment canvas and related support\n"
                                    + "    plugins/tables.jar"
                                    + "        Provides tabulated data support\n"
                                    + "\n"
                                    + "FILE FORMATS\n"
                                    + "    All file formats in BioLegato are supported through plugins.\n"
                                    + "    Below is a list of file formats supported by a default\n"
                                    + "    installation of BioLegato (with all standard plugins):\n"
                                    + "\n"
                                    + "        BioLegato flatfiles\n"
                                    + "        FastA files\n"
                                    + "        GDE flatfiles\n"
                                    + "        GDE format files\n"
                                    + "        GenBank files (2008 standard compliant)\n"
                                    + "\n"
                                    + "BUGS\n"
                                    + "    There are currently no known bugs in this version of BioLegato\n"
                                    + "\n"
                                    + "    Please report all bugs to: Graham Alvare <alvare@cc.umanitoba.ca>"
                                    + "\n"
                                    + "AUTHORS\n"
                                    + "    Dr. Brian Fristensky\n"
                                    + "    Department of Plant Science\n"
                                    + "    University of Manitoba\n"
                                    + "    Winnipeg, MB  Canada R3T 2N2\n"
                                    + "\n"
                                    + "    Email: frist@cc.umanitoba.ca\n"
                                    + "    Web: http://home.cc.umanitoba.ca/~frist\n"
                                    + "\n"
                                    + "    Graham Alvare\n"
                                    + "    Department of Plant Science\n"
                                    + "    University of Manitoba\n"
                                    + "    Winnipeg, MB  Canada R3T 2N2\n"
                                    + "\n"
                                    + "    Email: alvare@cc.umanitoba.ca\n"
                                    + "    Web: http://home.cc.umanitoba.ca/~alvare\n");
                            System.exit(0);
                        } else if ("version".equals(argument)
                                || "v".equals(argument)) {
                            ///////////////
                            //***********//
                            //* VERSION *//
                            //***********//
                            ///////////////
                            // show BioLegato's version
                            System.out.println("BioLegato v" + VERSION);
                            System.exit(0);
                        } else if ("debug".equals(argument)) {
                            /////////////
                            //*********//
                            //* DEBUG *//
                            //*********//
                            /////////////
                            // enable debug information
                            debug = true;
                        } else if ("plugins".equals(argument)) {
                            ///////////////
                            //***********//
                            //* PLUGINS *//
                            //***********//
                            ///////////////
                            // show BioLegato's loaded plugins
                            System.out.println("(Current plugins path: "
                                    + getProperty("plugins") + ")\n"
                                    + "-- listing plugins loaded --");
                            String[] pluginList =
                                    plugins.keySet().toArray(
                                    new String[0]);
                            for (String pluginName : pluginList) {
                                System.out.append("Plugin: ").append(pluginName).append("\n");
                            }
                            System.out.flush();
                            System.out.println("-- end of plugin list --");
                        } else if ("properties".equals(argument)) {
                            System.out.println("**********\n"
                                    + "PROPERTIES\n"
                                    + "**********");
                            //////////////////
                            //**************//
                            //* PROPERTIES *//
                            //**************//
                            //////////////////
                            // show BioLegato's current settings
                            properties.list(System.out);
                        } else {
                            BLMain.error("Unknown argument: " + rawarg,
                                    "processArgs");
                        }
                    }
                }
            }
        }
        // if debug is set, display the command
        if (BLMain.debug) {
            BLMain.message("Command line arguments read successfully",
                    "processArgs");
        }

        if ("true".equalsIgnoreCase(BLMain.getProperty("debug")) || (System.getenv("BL_DEBUG") != null
                && !"".equals(System.getenv("BL_DEBUG").trim())
                && !"false".equalsIgnoreCase(System.getenv("debug")))) {
            debug = true;
        }

////////////
//********//
//* MENU *//
//********//
////////////
        menu = new JMenuBar();

        addMenuHeading("File");

        ////////////////////////////////
        //****************************//
        //* READ IN THE CUSTOM MENUS *//
        //****************************//
        ////////////////////////////////

        /*************************************************************
         * ittearates through the Plugins searching for menu formats *
         *************************************************************/
        GDEMenuParser.loadGDEMenus();
        GDEMenuParser.loadMakemenus();

        ///////////////////////////////////////////
        //***************************************//
        //* ADD THE DEFAULT TRAILING MENU ITEMS *//
        //***************************************//
        ///////////////////////////////////////////

        /**************************
         * Add the "About" button *
         **************************/
        addMenuItem("Help", ABOUT_MENUITEM);	/* Event handler - open about it window */

        /*************************
         * Add the "Exit" button *
         *************************/
        addMenuItem("File", EXIT_MENUITEM);	/* Event handler - exit the program */

////////////////////////
//********************//
//* CONFIGURE WINDOW *//
//********************//
////////////////////////
        window = new JFrame(NAME);
        window.setJMenuBar(menu);

////////////////
//************//
//* CANVASES *//
//************//
////////////////
        // Load the canvas panes.
        // Populates the tabbed pane with all plugin canvases
        for (PluginWrapper plugin : plugins.values()) {
            if (plugin.isA(DataCanvas.class)) {
                try {
                    canvas = (DataCanvas) plugin.create();    // used for loading canvases (for testing proper object creation)

                    if (canvas.getName() != null
                            && canvas.getName().equalsIgnoreCase(getProperty("canvas"))) {
                        /////////
                        // end the loop early with the current canvas iff
                        // the current canvas is the one specified in the properties file
                        /////////
                        break;
                    }

                } catch (Throwable th) {
                    BLMain.error("error loading the plugin: " + plugin.getName(), "BLMain");
                    th.printStackTrace(System.err);
                }
            }
        }
        if (canvas != null) {
            window.add(canvas.display());
            for (File file : dataAdd) {
                canvas.readFile("", file);
            }
        } else {
            BLMain.error("could not load canvas!", "BLMain");
        }

///////////////
//***********//
//* DISPLAY *//
//***********//
///////////////

        // center and draw the frame on the screen
        window.pack();
        window.setVisible(true);
        window.setLocationRelativeTo(null);
        window.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    }

////////////////////////////
//************************//
//* PROPERTIES FUNCTIONS *//
//************************//
////////////////////////////
    /**
     * Retrieves individual settings for BioLegato.
     * This is used to obtain values of properties in BioLeagato
     *
     * @param property the property key to retrieve the value for
     * @return the property value corresponding to the key parameter
     */
    public static String getProperty(String property) {
        return properties.getProperty(property);
    }

//////////////////////
//******************//
//* DATA FUNCTIONS *//
//******************//
//////////////////////
    /**
     * Returns the current/selected data in the canvas.
     **
     * @return the current data for usage by commands
     */
    public static DataCanvas getCanvas() {
        return canvas;
    }

////////////////////////
//********************//
//* WINDOW FUNCTIONS *//
//********************//
////////////////////////
    /**
     * Provides access for other classes to the main program window JFrame object.
     **
     * @return the JFrame object corresponding to the main program window.
     */
    public static JFrame getJFrame() {
        return window;
    }

    /**
     * Adds a menu heading (JMenu) to our menu (BLMenu).
     **
     * @param name the name of the menu heading
     * @return either the JMenu which was added or the JMenu that corresponds to the existing menu heading.
     */
    public static JMenu addMenuHeading(String name) {
        JMenu heading = menuHeadings.get(name);	// the JMenu object to add

        // check if the heading already exists in the menu. (null indicates that the menu heading is new)
        if (heading == null) {
            // create a new menu heading object
            heading = new JMenu(name);

            // set the mnemonic
            if (name != null && name.length() >= 1
                    && ((name.charAt(0) >= 'a' && name.charAt(0) <= 'z')
                    || (name.charAt(0) >= 'A' && name.charAt(0) <= 'Z'))) {
                heading.setMnemonic(name.charAt(0));
            }

            // add the heading
            menuHeadings.put(name, heading);
            menu.add(heading);
        }

        // return the menu heading object
        return heading;
    }

    /**
     * Adds a menu heading (JMenu) to our menu (BLMenu).
     **
     * @param order the position to place the menu tag
     * @param name the name of the menu heading
     * @return either the JMenu which was added or the JMenu that corresponds to the existing menu heading.
     */
    public static JMenu addMenuHeading(int order, String name) {
        JMenu heading = addMenuHeading(name);	// the hading to add the item to.

        // ensure that the menu heading is in the correct order
        if (menu.getComponentIndex(heading) != order) {
            menu.remove(heading);
            menu.add(heading, order);
        }
        return heading;
    }

    /**
     * Adds a menu item (JMenuItem) to a menu heading (JMenu) within the window (ProgramWindow)
     * <p>If the heading which was entered does not exist, this function will create it;
     *	hence why addMenuHeading returns the heading corresponding to the name entered if a
     *	menu heading with that name already exists.</p>
     **
     * @param location is the location in the menu to insert the item at
     * @param headingName is the name of the menu heading to add the menu item to.
     * @param menuItem is the menu item to add to the heading.
     */
    public static void addMenuItem(int location, String headingName, JMenuItem menuItem) {
        // enforce that the menu heading exists and then add the menu item to it
        addMenuHeading(headingName).insert(menuItem, location);
    }

    /**
     * Adds a menu item (JMenuItem) to a menu heading (JMenu) within the window (ProgramWindow)
     * <p>If the heading which was entered does not exist, this function will create it;
     *	hence why addMenuHeading returns the heading corresponding to the name entered if a
     *	menu heading with that name already exists.</p>
     *
     * @param headingName is the name of the menu heading to add the menu item to.
     * @param menuItem is the menu item to add to the heading.
     */
    public static void addMenuItem(String headingName, JMenuItem menuItem) {
        // enforce that the menu heading exists and then add the menu item to it
        addMenuHeading(headingName).add(menuItem);
    }

/////////////////////////
//*********************//
//* GENERAL FUNCTIONS *//
//*********************//
/////////////////////////
    /**
     * Runs simple shell commands.
     * Reroutes all output to the console.
     **
     * @param cmd the command string to run
     * @param data the data to use as standard input (System.in)
     */
    public static void shellCommand(String cmd, String data) {
        int result = -65535;			    // the result of running the command
        Process process = null;                     // the process object obtained on execution
        StringBuilder message = new StringBuilder();	// used for printing debug information
        String[] execute = new String[]{cmd};           // default case - run the command by itself
        final int BUFF_SIZE = 1000;

        /* Ensures that the command will be executed properly as a shell command
         * <p>This function generates a command list for execution.  The command list will contain
         *	the appropriate shell for the current operating system, followed by the "execution-argument",
         *	(whatever flag is required to tell the shell that the rest of the commandline should be executed
         *	by the shell), followed by the command to execute (the variable cmd)</p>
         * <p>Operating system values obtained from http://lopica.sourceforge.net/os.html</p>
         */
        // builds the execution array
        if (properties.getProperty("shell.name") != null) {
            // if there is a shell for the current operating system, execute the command through the shell
            if (properties.getProperty("shell.parameter") != null) {
                execute =
                        new String[]{properties.getProperty("shell.name"),
                            properties.getProperty("shell.parameter"), cmd};
            } else {
                execute = new String[]{properties.getProperty("shell.name"),
                            cmd};
            }
        }

        // if debug is set, display the command
        if (BLMain.debug) {
            for (String parameter : execute) {
                // NOTE: append is faster than + or concat operators
                message.append(parameter).append(" ");
            }
            // relay the command string the message system
            BLMain.message(message.toString(), "safeExecute");
        }

        // obtain the process object for the command
        try {
            process = Runtime.getRuntime().exec(execute);

            // ensure the process object is not null
            if (process != null) {
                // passes any data to the program via standard in
                if (data != null) {
                    (new OutputStreamWriter(process.getOutputStream())).write(
                            data);
                }

                new Thread(new StreamCopier(BUFF_SIZE, process.getInputStream(), System.out)).start();
                new Thread(new StreamCopier(BUFF_SIZE, process.getErrorStream(), System.out)).start();

                // display the command's result if debug is enabled
                if (BLMain.debug) {
                    BLMain.message("BioLegato: Command executed successfully, returned: "
                            + process.waitFor(), "shellCommand");
                } else {
                    process.waitFor();
                }
            }
        } catch (Throwable e) {
            // if there are any errors, print them to the error prompt
            BLMain.error("BioLegato: error executing command: " + cmd,
                    "shellCommand");
            e.printStackTrace(System.err);
        }
    }

    /**
     * Sends an error message to BioLegato's standard err.
     **
     * @param message the error message to send.
     * @param location the location the error occurred.
     */
    public static void error(String message, String location) {
        // print the error to the error stream
        System.err.println(NAME + ((location != null && !"".equals(location.trim()))
                ? " (" + location + ")" : "") + ": ERROR --- " + message);
    }

    /**
     * Sends an warning message to BioLegato's standard err.
     **
     * @param message the warning message to send.
     * @param location the location the error occurred.
     */
    public static void warning(String message, String location) {
        // prints the warning to BioLegato's error stream
        System.err.println(NAME + ((location != null && !"".equals(location.trim()))
                ? " (" + location + ")" : "") + ": WARNING --- " + message);
    }

    /**
     * Sends a message to BioLegato's standard out.
     **
     * @param message the message to send.
     * @param location the location the message was sent from.
     */
    public static void message(String message, String location) {
        // prints the warning to BioLegato's error stream
        System.out.println(NAME + ((location != null && !"".equals(location.trim()))
                ? " (" + location + ")" : "") + ": " + message);
    }

    /**
     * Checks if a character array is all digits.
     **
     * @param test the character array to test.
     * @return true if the array only contains digits.
     */
    public static boolean testNumber(String test) {
        return Pattern.matches("^-?\\d+$", test);
    }

    /**
     * Divide and conquer algorithm to calculate the number of digits for a given
     * number
     **
     * @param n the number to calculate the digit content
     * @returns the number of digits for number 'n'
     */
    public static int numberDigits(int n) {
        if (n < 100000) {
            // 5 or less
            if (n < 100) {
                // 1 or 2
                if (n < 10) {
                    return 1;
                } else {
                    return 2;
                }
            } else {
                // 3 or 4 or 5
                if (n < 1000) {
                    return 3;
                } else {
                    // 4 or 5
                    if (n < 10000) {
                        return 4;
                    } else {
                        return 5;
                    }
                }
            }
        } else {
            // 6 or more
            if (n < 10000000) {
                // 6 or 7
                if (n < 1000000) {
                    return 6;
                } else {
                    return 7;
                }
            } else {
                // 8 to 10
                if (n < 100000000) {
                    return 8;
                } else {
                    // 9 or 10
                    if (n < 1000000000) {
                        return 9;
                    } else {
                        return 10;
                    }
                }
            }
        }
    }

    /**
     * Replaces $VARIABLE name with environment variables.
     * This function provides functionality similar to BASH.
     *
     * NOTE: if the variable is $BL_HOME, and this is not set, 
     *	the variable will default to the current working directory.
     **
     * @param change the string to modify.
     * @return the modified string.
     */
    public static String envreplace(String change) {
        int start = 0;
        int end = -1;
        String replace = null;
        String variable = "";

        if (change != null) {
            while ((start = change.indexOf('$', start)) >= start && start > -1) {
                for (end = start + 1; end < change.length() && (change.charAt(end) == '_' || Character.isLetter(change.charAt(end))); end++) {
                    /* ITTERATE */
                }

                // get the information to perform the string replacement.
                variable = change.substring(start + 1, end);
                replace = System.getenv(variable);

                // ensure BL_HOME is set properly.
                if (variable.equalsIgnoreCase("BL_DIR")) {
                    replace = BLMain.PROGRAM_DIR;
                }

                // ensure BL_HOME is set properly.
                if (variable.equalsIgnoreCase("BL_HOME")) {
                    if (replace == null || "".equals(replace.trim()) || !new File(replace).exists()) {
                        replace = BLMain.CURRENT_DIR;
                    } else {
                        replace = new File(replace).getAbsolutePath();
                    }
                }

                // perform the string replacement.
                if (replace != null) {
                    change = change.substring(0, start) + replace + change.substring(end);
                } else {
                    start++;
                }
            }
        }
        return change;
    }

    /**
     * Gets the present working directory "PWD" for BioLegato (this does NOT change CURRENT_DIR!)
     * This value can be used to cache which directory file dialog boxes should use as their
     * "current working directory"
     **
     * @return the current value of "PWD"
     */
    public static File getCurrentPWD() {
        return currentPWD;
    }

    /**
     * Sets the present working directory "PWD" for BioLegato (this does NOT change CURRENT_DIR!)
     * This value can be used to cache which directory file dialog boxes should use as their
     * "current working directory"
     **
     * @param newPWD the new directory to use as the "PWD"
     */
    public static void setCurrentPWD(File newPWD) {
        currentPWD = newPWD;
    }

//////////////////
//**************//
//* DEPRICATED *//
//**************//
//////////////////
    /**
     * Tests if a character is a digit or not.
     * Digits are from 0-9, a-z, and A-Z
     **
     * @param test the character to test.
     * @return true if the character is a digit, otherwise false.
     */
    public static boolean isDigit(char test) {
        return ((test >= '0' && test <= '9') || (test >= 'a' && test <= 'z') || (test >= 'A' && test <= 'Z'));
    }
}
