/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.canvas;

import org.biolegato.gdesupport.data.GDESequenceWindow;
import java.awt.Color;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.event.ListDataListener;
import org.biolegato.gdesupport.data.Seq;
import org.biolegato.gdesupport.data.ClipboardWrapper;
import org.biolegato.gdesupport.files.DataFormat;
import org.biolegato.main.BLMain;
import org.biolegato.gdesupport.data.Dataset;

/**
 * The JList of sequences in the GDE canvas.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEList extends JList implements MouseListener, GDECanvasObject {

    /**
     * The last time the mouse was clicked (used for double click detection)
     */
    private long lastClickTime = -1000;
    /**
     * The entry selected on the last click
     */
    private int lastClickEntry = -1;
    /**
     * Self reference.
     */
    private final GDEList list = this;
    /**
     * The right click menu for the GDE sequence list.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * Used for the relationship between the data model and the list
     */
    protected Dataset datamodel;
    /**
     * The list of listeners to the data in the list.
     */
    private List<ListDataListener> listListeners = new ArrayList<ListDataListener>();
    /**
     * The menu item "Select group"
     */
    public final AbstractAction selectGroupAction = new AbstractAction("Select group") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            int[] group = null;

            for (int index : getSelectedIndices()) {
                // getLine the currently selected sequence
                group = datamodel.getgroup(datamodel.getLine(index).getGroupID());
                if (group != null) {
                    // set the selection indicies
                    setSelectedIndices(group);
                }
            }
        }
    };
    /**
     * The menu item "Get info..."
     */
    public final AbstractAction getInfoAction = new AbstractAction("Get info...") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_G));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            if (!isSelectionEmpty()) {
                new GDESequenceWindow(datamodel, BLMain.getJFrame(), getSelectedIndices());
            }
        }
    };
    /**
     * The menu item "Delete"
     */
    private final JMenuItem deleteMenuItem = new JMenuItem(new AbstractAction("Delete") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_D));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            deleteSelection();
        }
    });
    /**
     * The menu item "Select All"
     */
    public final AbstractAction selectAllAction = new AbstractAction("Select All") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_A));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            list.setSelectionInterval(0, list.getModel().getSize() - 1);
            list.requestFocus();
        }
    };
    /**
     * The abstract action "Select sequence by name"
     */
    public final AbstractAction selectByNameAction = new AbstractAction("Select sequence by name") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            list.requestFocus();
            list.requestFocusInWindow();
            new SelectByNameWindow(BLMain.getJFrame(), list);
        }
    };
    /**
     * Action for grouping sequences in the canvas
     */
    public final AbstractAction groupAction = new AbstractAction("Group") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed(ActionEvent e) {
            datamodel.group((Seq[]) list.getSelectedValues());
        }
    };
    /**
     * Action for ungrouping sequences in the canvas
     */
    public final AbstractAction ungroupAction = new AbstractAction("Ungroup") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed(ActionEvent e) {
            datamodel.ungroup((Seq[]) list.getSelectedValues());
        }
    };
    /**
     * Used in serialization.
     */
    private static final long serialVersionUID = 7526472295622777009L;

    /**
     * Constructs a new GDEList
     */
    public GDEList(GDECanvas canvas, Dataset datamodel) {
        super(datamodel);

        // initialize variables
        this.datamodel = datamodel;

        // build list popupmenu
        popup.add(new JMenuItem(canvas.cutAction));
        popup.add(new JMenuItem(canvas.copyAction));
        popup.add(new JMenuItem(canvas.pasteAction));
        popup.add(deleteMenuItem);
        popup.add(new JMenuItem(groupAction));
        popup.add(new JMenuItem(ungroupAction));
        popup.add(new JMenuItem(selectAllAction));
        popup.add(new JMenuItem(selectGroupAction));
        popup.add(new JMenuItem(selectByNameAction));
        popup.add(new JMenuItem(getInfoAction));

        // add listeners
        addMouseListener(list);

        // set display properties
        setForeground(Color.BLACK);
        setBackground(new Color(255, 255, 240));
        setSize(getPreferredSize());
        setFont(canvas.currentFont);
    }

    /**
     * Checks for double clicks.  On a double click, this method opens
     * up a sequence properties window.
     *
     * @param event the mouse event to check for the double click.
     */
    public void mouseClicked(MouseEvent event) {
        // if double click select the entire group
        if (event.getClickCount() > 2 || (getSelectedIndex() == lastClickEntry
                && event.getWhen() - lastClickTime < BLMain.DOUBLE_CLICK_TIME)) {
            selectGroupAction.actionPerformed(null);
        }
        lastClickTime = event.getWhen();
        lastClickEntry = getSelectedIndex();
    }

    /**
     * Checks for right clicks.  On a right click, this method opens up a popup menu
     *
     * @param event the mouse event to check for the right click.
     */
    public void mousePressed(MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX() - getX(), event.getY() - getY());
        }
    }

    /**
     * Checks for right clicks.  On a right click, this method opens up a popup menu
     *
     * @param event the mouse event to check for the right click.
     */
    public void mouseReleased(MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX() - getX(), event.getY() - getY());
        }
    }

    /**
     * Currently does nothing
     *
     * @param event ignored.
     */
    public void mouseEntered(MouseEvent event) {
    }

    /**
     * Currently does nothing
     *
     * @param event ignored.
     */
    public void mouseExited(MouseEvent event) {
    }

    /**
     * Pastes content into the current Editable object.
     */
    public void paste() {
        int row = datamodel.getSize();
        Seq[] clipboard = ClipboardWrapper.getClipboard();

        if (clipboard != null) {
            if (getSelectedIndex() >= 0) {
                row = getSelectedIndex() + 1;
            }

            for (Seq seq : clipboard) {
                datamodel.addSequence(row, seq);
                row++;
            }
        }
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase() {
        String data = null;

        for (int sequenceNumber : getSelectedIndices()) {
            data = datamodel.getLine(sequenceNumber).getSequence().toString();

            if (data.length() > 0) {
                if (Character.isUpperCase(data.charAt(0))) {
                    data.toLowerCase();
                } else {
                    data.toUpperCase();
                }
            }
            datamodel.getLine(sequenceNumber).getSequence().replace(0, data.length(), data);
        }
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public void setFont(Font font) {
        super.setFont(font);
        setFixedCellHeight(getFontMetrics(font).getHeight());
        repaint();
    }

    /**
     * Displays a window to edit the currently selected sequence's properties
     */
    public void deleteSelection() {
        final int[] deletionZone = getSelectedIndices();

        datamodel.removeSequences(deletionZone);
    }

    /**
     * Returns the currently selected data in the list.
     *
     * @return the current data for usage by commands
     */
    public void copyClipboard() {
        Seq seqAdd = null;                                     // the result of the current call to getSequence
        int count = 0;
        int countseq = 0;
        int[] indexlist = getSelectedIndices();
        Seq[] sequencelist = new Seq[indexlist.length];

        // itterate through the line numbers
        for (count = 0; count < indexlist.length; count++) {
            seqAdd = datamodel.getLine(indexlist[count]);

            // ensure that we do not add any null results
            if (seqAdd != null) {
                sequencelist[countseq] = seqAdd;
                countseq++;
            }
        }
        // ensure the proper array length
        if (countseq < indexlist.length) {
            Seq[] temp = new Seq[countseq];
            System.arraycopy(sequencelist, 0, temp, 0, countseq);
            sequencelist = temp;
        }
        ClipboardWrapper.setClipboard(sequencelist);
    }

    public void writeOut(DataFormat format, Appendable dest) throws IOException {
        Seq line;
        
        // translate and write the file.
        for (int lineNumber : getSelectedIndices()) {
            line = datamodel.getLine(lineNumber);
            format.translateTo(dest, line, 0, line.getSequence().length());
        }
    }

}
