package org.biolegato.gdesupport.canvas;

/*
 * GDETextArea.java
 *
 * Created on August 18, 2008, 11:10 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import javax.swing.JComponent;
import org.biolegato.gdesupport.data.Dataset;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.io.IOException;
import java.util.Arrays;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import org.biolegato.gdesupport.data.Seq;
import org.biolegato.gdesupport.data.ClipboardWrapper;
import org.biolegato.gdesupport.data.ColourMask;
import org.biolegato.gdesupport.files.DataFormat;
import org.biolegato.main.BLMain;

/**
 * A general canvas with more functionality support than JTextArea.
 *
 * This canvas was originally created to support the rectangular selection model,
 * and has since supported rectangular selections, different colour schemes,
 * sequence documents, and many other features.
 * Please add more generic feature support as necessary.
 *
 * Also if you wish to add any functionality or handling specific to a plugin,
 * please feel free to subclass this textarea.
 **
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDETextArea extends JComponent implements ListDataListener, GDECanvasObject, KeyListener, MouseMotionListener, MouseListener {

    /**
     * The height of each row in the textarea
     */
    protected int rowHeight = 1;
    /**
     * The width of each column in the textarea
     */
    protected int columnWidth = 1;
    /**
     * The canvas associtated with the textarea.
     */
    private GDECanvas canvas = null;
    /**
     * If true new text will overwrite existing text.
     */
    protected boolean insertMode = false;
    /**
     * Used to track the shift key.
     * If this is false, cancel the current selection on moving using the direction arrows.
     * If this is true, stretch the current selection on moving using the direction arrows.
     */
    protected boolean selectionMove = false;
    /**
     * Used to track mouse holds and releases.
     * If this is false, cancel the current selection on any mouse click.
     * If this is true, stretch the current selection on mouse movement.
     */
    protected boolean selectionMouse = false;
    /**
     * The current row of the caret
     */
    protected int row = 0;
    /**
     * The current column of the caret.
     */
    protected int column = 0;
    /**
     * The start column of the selection (no selection if sx = column)
     */
    private int sx = 0;
    /**
     * The start row of the selection (no selection if sy = row)
     */
    private int sy = 0;
    /**
     * The minimum X co-ordinate for the current selection (cached for faster drawing)
     *
     * NOTE: this should only be modified by changePosition!!!
     */
    private int minsx = 0;
    /**
     * The minimum Y co-ordinate for the current selection (cached for faster drawing)
     *
     * NOTE: this should only be modified by changePosition!!!
     */
    private int minsy = 0;
    /**
     * The maximum X co-ordinate for the current selection (cached for faster drawing)
     *
     * NOTE: this should only be modified by changePosition!!!
     */
    private int maxsx = 0;
    /**
     * The maximum Y co-ordinate for the current selection (cached for faster drawing)
     *
     * NOTE: this should only be modified by changePosition!!!
     */
    private int maxsy = 0;
    /**
     * The right click menu for the text area.
     */
    protected JPopupMenu popup = new JPopupMenu();
    /**
     * Used for the relationship between the data model and the text area
     */
    protected Dataset datamodel;
    /**
     * Self-reference for inner classes.
     */
    public final GDETextArea blTextAreaSelf = this;
    /**
     * Stores the length of the longest line in the textarea
     */
    private int longestline = 0;
    /**
     * The FOREG colour of selected text
     */
    public static final Color SELECTFG = Color.WHITE;
    /**
     * The BACKG colour of selected text
     */
    public static final Color SELECTBG = Color.BLUE;
    /**
     * This constant is used for serialization purposes.
     */
    public static final long serialVersionUID = 7526472295622777004L;

    /**
     * Creates a new instance of GDETextArea
     **
     * @param canvas the parent canvas of the text area
     * @param datamodel the data model to view the text from
     */
    public GDETextArea (GDECanvas canvas, Dataset datamodel) {
        // import all applicable constructor parameters
        this.datamodel = datamodel;

        // link the text area to its canvas
        this.canvas = canvas;

        // add the listeners
        datamodel.addListDataListener(blTextAreaSelf);
        addKeyListener(blTextAreaSelf);
        addMouseListener(blTextAreaSelf);
        addMouseMotionListener(blTextAreaSelf);

        // create popup menu
        popup.add(new JMenuItem(canvas.cutAction));
        popup.add(new JMenuItem(canvas.copyAction));
        popup.add(new JMenuItem(canvas.pasteAction));

        // set display
        setFont(canvas.currentFont);
        setForeground(Color.BLACK);
        setBackground(Color.WHITE);
        setFocusTraversalKeysEnabled(false);
        refreshSize();
        setDoubleBuffered(true);
        repaint();
    }

///////////////////////
//*******************//
//* DISPLAY METHODS *//
//*******************//
///////////////////////
    /**
     * Used to obtain all of the text selected within the document.
     */
    public void copyClipboard () {
        // TODO: getSelectedText
        int lineLength;
        StringBuffer originalSequence = null;
        StringBuffer currentSequence = new StringBuffer();
        Seq current = null;
        int countseq = 0;
        final int ylength = maxsy - minsy + 1;
        Seq[] result = null;

        if (!isSelectionEmpty()) {
            result = new Seq[ylength];
            for (int count = minsy; count <= maxsy; count++) {
                lineLength = getLineLength(count);
                if (minsx < lineLength) {
                    current = datamodel.getLine(count);
                    if (current != null) {
                        if (!isSelectionEmpty()) {
                            int xstart = minsx;
                            int xlength = (originalSequence.length() > maxsx ? maxsx
                                    : originalSequence.length()) - xstart + 1;

                            if (xlength > 0) {
                                currentSequence.append(originalSequence.subSequence(xstart, xlength));
                            }

                            current = new Seq(current.getType(), current.getName(),
                                    currentSequence, current.getDirection(),
                                    current.getTopology(), current.getStrandedness());
                        } else {
                            current = (Seq) current.clone();
                        }
                        result[countseq] = current;
                        countseq++;
                    }
                }
            }
            // ensure the proper array length
            if (countseq < ylength) {
                Seq[] temp = new Seq[countseq];
                System.arraycopy(result, 0, temp, 0, countseq);
                result = temp;
            }
        } else {
            result = new Seq[0];
        }
        ClipboardWrapper.setClipboard(result);
    }

    /**
     * Paints the current textarea (this uses the current clip bounds to determine the area to paint.
     *
     * @param gfx the graphics instance to paint the window to.
     */
    @Override
    public void paintComponent (Graphics gfx) {
        try {
            final int maxrows = datamodel.getSize();
            final Rectangle area = gfx.getClipBounds();   // getLine the area to paint

            // convert the area of the rectangle into columns and rows
            final int startcol = area.x / columnWidth;
            final int startrow = area.y / rowHeight;
            final int collength = (area.width / columnWidth) + 2;
            final int numrows = (area.height / rowHeight) + 2;
            final int endcol = startcol + collength;
            final int endrow = startrow + (area.height / rowHeight) + 2;
            final int stoprow = (endrow < maxrows ? endrow : maxrows - 1);

            // print adjustment variables
            final int fontcenterXmod = columnWidth / 2;
            final int fontcenterYmod = -5;

            // the position to begin drawing the rectangle at
            final int xstartfill = startcol * columnWidth;
            final int ystartfill = startrow * rowHeight;

            // the position to begin drawing characters at
            final int xstart = xstartfill + fontcenterXmod;
            final int ystart = ystartfill + fontcenterYmod + rowHeight;

            // getLine the selection start position
            final int startSelectX = (columnWidth * minsx) + fontcenterXmod;
            final int endSelectX = (columnWidth * maxsx) + fontcenterXmod;
            final int selectXlength = endSelectX - startSelectX;

            final int cursorX = column * columnWidth;
            final int cursorY = row * rowHeight;

            final int minsxcol = Math.max(0, minsx - startcol);
            final int maxsxcol = Math.max(0, maxsx - startcol);
            final int sxlength = maxsxcol - minsxcol;

            int drawrow;
            int drawend;
            int drawcolend;
            int sequenceLength;
            char[] print = new char[collength];
            Seq currentLine = null;
            ColourMask mask = null;

            // print the background
            gfx.setFont(getFont());
            gfx.setColor(getBackground());
            gfx.fillRect(startcol * columnWidth, startrow * rowHeight,
                    collength * columnWidth, numrows * rowHeight);

            // paint the selection background (the blue part)
            if (maxsx > startcol && minsx < endcol && maxsy > startrow
                    && minsy < stoprow) {
                gfx.setColor(SELECTBG);
                gfx.fillRect(startSelectX, Math.max(startrow, minsy) * rowHeight - (fontcenterYmod / 2), selectXlength,
                        Math.min(stoprow, maxsy - minsy + 1) * rowHeight);
            }

            // print each sequence
            for (int rowcount = startrow, currenty = ystart;
                    rowcount <= stoprow; rowcount++, currenty += rowHeight) {

                // getLine the data to print
                currentLine = datamodel.getLine(rowcount);
                sequenceLength = currentLine.getSequence().length();
                drawend = (endcol < sequenceLength ? endcol : sequenceLength);
                drawcolend = drawend - startcol;

                if (drawcolend > 0) {
                    currentLine.getSequence().getChars(startcol, drawend, print, 0);

                    // use the sequence specific colour masks if available
                    mask = currentLine.getMask();

                    // check whether the segment of the current row
                    // to be drawn contains a portion that is selected
                    if (rowcount >= minsy && rowcount <= maxsy) {
                        // since the line contains selected text,
                        // check whether the selection starts before
                        // or after the starting position of the area
                        // of the screen to draw
                        if (minsx > startcol) {
                            // since the selection starts before the start
                            // position of the area of the screen to draw
                            // draw both the non-selected text preceding
                            // and the selected text
                            if (mask != null) {
                                mask.drawString(gfx, print, xstart, currenty, 0, minsxcol);
                            } else {
                                gfx.setColor(ColourMask.FOREG);
                                gfx.drawChars(print, 0, minsxcol, xstart, currenty);
                            }
                        }
                        if (maxsx > startcol && minsx < drawend) {
                            // set the foreground colour and paint the text
                            gfx.setColor(SELECTFG);
                            gfx.drawChars(print, minsxcol, (maxsxcol < drawcolend ? sxlength : drawcolend - minsxcol), startSelectX, currenty);
                        }
                        if (maxsxcol < drawcolend) {
                            if (mask != null) {
                                mask.drawString(gfx, print, endSelectX, currenty, maxsxcol, drawcolend - maxsxcol);
                            } else {
                                gfx.setColor(ColourMask.FOREG);
                                gfx.drawChars(print, maxsxcol, drawcolend - maxsxcol, endSelectX, currenty);
                            }
                        }
                    } else {
                        if (mask != null) {
                            mask.drawString(gfx, print, xstart, currenty, 0, drawcolend);
                        } else {
                            gfx.setColor(ColourMask.FOREG);
                            gfx.drawChars(print, 0, drawcolend, xstart, currenty);
                        }
                    }
                }
            }
            // draw the cursor.
            if (hasFocus() && gfx.hitClip(cursorX, cursorY, 1, rowHeight)) {
                gfx.setColor(getForeground());
                gfx.drawLine(cursorX, cursorY, cursorX, cursorY + rowHeight);
            }
        } catch (Throwable th) {
            // don't halt printing other objects on crash
            th.printStackTrace(System.err);
        }
    }

    /**
     * Updates the font for the canvas (ensures repaint)
     *
     * @param font the new font to handle.
     */
    @Override
    public final void setFont (Font font) {
        super.setFont(font);
        columnWidth = getFontMetrics(font).charWidth('G');
        rowHeight = getFontMetrics(font).getHeight();
        refreshSize();
        repaint();
    }

////////////////////////
//********************//
//* KEYBOARD METHODS *//
//********************//
////////////////////////
    /**
     * Processes the typing of keys within the text area
     *
     * @param event the KeyEvent for the key typed
     */
    public void keyTyped (KeyEvent event) {
        boolean canInsert = true;

        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                case KeyEvent.VK_DELETE:
                    break;
                case KeyEvent.VK_ENTER:
                    // now only changes the row (same effect as downkey, except clears selection)
                    if (row + 1 < datamodel.getSize()) {
                        changePosition(false, column, row + 1);
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    break;
                default:
                    if (!isSelectionEmpty()) {
                        gdeleteSection();
                    } else if (insertMode && datamodel.getSize() > 0
                            && getLineLength(0) > 0) {
                        canInsert = delete(column, row, 1, 0);
                    }
                    if (isSelectionEmpty() || !insertMode || canInsert) {
                        insert(column, row, new char[]{event.getKeyChar()});
                        changePosition(false, column + 1, row);
                    }
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace(System.err);
        }
        event.consume();
    }

    /**
     * Processes key presses within the text area
     *
     * @param event the KeyEvent for the key pressed
     */
    public void keyPressed (KeyEvent event) {
        int backspacecol = 0;

        try {
            switch (event.getKeyChar()) {
                case KeyEvent.VK_BACK_SPACE:
                    if (!isSelectionEmpty()) {
                        gdeleteSection();
                    } else if (row > 0 || column > 0) {
                        if (column <= 0 && row > 0) {
                            backspacecol = getLineLength(row - 1);
                        }
                        if (column - 1 >= 0) {
                            if (delete(column - 1, row, 1, 0)) {
                                if (column <= 0 && row > 0) {
                                    changePosition(false, backspacecol, row - 1);
                                } else {
                                    changePosition(false, column - 1, row);
                                }
                            }
                        }
                    }
                    break;
                case KeyEvent.VK_DELETE:
                    if (!isSelectionEmpty()) {
                        gdeleteSection();
                    } else if (datamodel.getSize() > 0
                            && getLineLength(0) > 0) {
                        delete(column, row, 1, 0);
                    }
                    break;
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            selectionMove = true;
                            break;
                        case KeyEvent.VK_LEFT:
                            if (column > 0) {
                                changePosition(selectionMove, column - 1, row);
                            } else if (row > 0) {
                                changePosition(selectionMove, getLineLength(row), row - 1);
                            }
                            break;
                        case KeyEvent.VK_RIGHT:
                            if (column + 1 <= getLineLength(row)) {
                                changePosition(selectionMove, column + 1, row);
                            } else if (row + 1 < datamodel.getSize()) {
                                changePosition(selectionMove, 0, row + 1);
                            }
                            break;
                        case KeyEvent.VK_UP:
                            if (row > 0) {
                                changePosition(selectionMove, column, row - 1);
                            }
                            break;
                        case KeyEvent.VK_DOWN:
                            if (row + 1 < datamodel.getSize()) {
                                changePosition(selectionMove, column, row + 1);
                            }
                            break;
                        case KeyEvent.VK_HOME:
                            changePosition(selectionMove, 0, row);
                            break;
                        case KeyEvent.VK_END:
                            changePosition(selectionMove, getLineLength(row), row);
                            break;
                        case KeyEvent.VK_COPY:
                            canvas.copyAction.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST, "copy"));
                            break;
                        case KeyEvent.VK_CUT:
                            canvas.cutAction.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST, "copy"));
                            break;
                        case KeyEvent.VK_PASTE:
                            canvas.pasteAction.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST, "copy"));
                            break;
                        case KeyEvent.VK_INSERT:
                            insertMode = !insertMode;
                            canvas.insertionMode(insertMode);
                            break;
                        default:
                            if (BLMain.debug) {
                                BLMain.warning("Unhandled key pressed --- getKeyChar() = " + ((int) event.getKeyChar()) + "\tgetKeyCode() = " + event.getKeyCode(), "BLTextArea");
                            }
                            break;
                    }
                    if (column >= getLineLength(row)) {
                        changePosition(selectionMove, getLineLength(row), row);
                    }
                    break;
                default:
                    gdeleteSection();
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace(System.err);
        }
        event.consume();
    }

    /**
     * Processes key releases within the text area
     **
     * @param event the KeyEvent for the key released
     */
    public void keyReleased (KeyEvent event) {
        try {
            switch (event.getKeyChar()) {
                case KeyEvent.CHAR_UNDEFINED:
                    switch (event.getKeyCode()) {
                        case KeyEvent.VK_SHIFT:
                            selectionMove = false;
                            break;
                        default:
                            break;
                    }
                    break;
                default:
                    break;
            }
        } catch (Throwable e) {
            e.printStackTrace(System.err);
        }
        event.consume();
    }

/////////////////////
//*****************//
//* MOUSE METHODS *//
//*****************//
/////////////////////
    /**
     * Handles mouse clicks.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set, in which case it stretches the selection).
     *
     * @param event the MouseEvent object corresponding to the click.
     */
    public void mouseClicked (MouseEvent event) {
        if (!event.isPopupTrigger()) {
            // calculate the row and column numbers from the X and Y co-ordinates.
            changePosition(selectionMove, Math.round(event.getX() / columnWidth), Math.round(event.getY() / rowHeight));

            // request focus within the current window
            requestFocus();
            requestFocusInWindow();
        }
    }

    /**
     * Handles mouse button presses.
     * This method sets the current caret and clears any selections
     * (unless selectionMove is set).
     *
     * @param event the MouseEvent object corresponding to the press.
     */
    public void mousePressed (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(this, event.getX(), event.getY());
        }
    }

    /**
     * Handles mouse button releases.
     *
     * @param event the MouseEvent object corresponding to the release.
     */
    public void mouseReleased (MouseEvent event) {
        if (event.isPopupTrigger()) {
            popup.show(event.getComponent(), event.getX(), event.getY());
        } else {
            selectionMouse = false;
        }
    }

    /**
     * Handles the mouse entering the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the enter.
     */
    public void mouseEntered (MouseEvent event) {
    }

    /**
     * Handles the mouse exiting the component.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the exit.
     */
    public void mouseExited (MouseEvent event) {
    }

    /**
     * Handles mouse drags.
     * Updates the selection and caret on drags.
     *
     * @param event the MouseEvent object corresponding to the drag.
     */
    public void mouseDragged (MouseEvent event) {
        if (!event.isPopupTrigger() || !isSelectionEmpty()) {
            // calculate the row and column numbers from the X and Y co-ordinates and update the position
            changePosition(selectionMouse || selectionMove, Math.round(event.getX() / columnWidth), Math.round(event.getY() / rowHeight));

            selectionMouse = true;
            requestFocus();
            requestFocusInWindow();
        }
    }

    /**
     * Handles mouse movements.
     * Currently does nothing
     *
     * @param event the MouseEvent object corresponding to the movement.
     */
    public void mouseMoved (MouseEvent event) {
    }

    /**
     * Refreshes the size of the textarea (for scroll size purposes)
     */
    protected final void refreshSize () {
        final int width = (longestline + 1) * columnWidth;
        final int height = rowHeight * datamodel.getSize();

        setSize(width, height);
        setPreferredSize(getSize());
        revalidate();
    }

/////////////////////////////////////
//*********************************//
//* EXTERNAL MANIPULATION METHODS *//
//*********************************//
/////////////////////////////////////
    /**
     * Pastes content into the current Editable object.
     */
    public void paste () {
        Seq current;
        Seq[] data = ClipboardWrapper.getClipboard();
        Seq[] temp;
        StringBuffer sequence;
        char[] text;
        int minlength = Math.min(data.length, datamodel.getSize());

        deleteSelection();
        for (int count = row; count < minlength; count++) {
            current = datamodel.getLine(count);
            sequence = current.getSequence();
            text = new char[sequence.length()];
            sequence.getChars(0, sequence.length(), text, 0);

            if (!isProtectionsOn(current.getType(), current.getProtectAlignment(),
                    current.getProtectAmbiguous(), current.getProtectUnambiguous(),
                    text, 0, text.length)) {
                sequence.insert(column, text);
            }
        }
        if (minlength < data.length) {
            temp = new Seq[data.length - minlength];
            System.arraycopy(data, minlength, temp, 0, data.length - minlength);
            datamodel.addSequences(minlength, Arrays.asList(temp));
        }
    }

    /**
     * Changes the case of the currently selected sequence
     * (if the sequence is of inconsistent case, the case of the entire sequence
     * is changed to the opposite case of the first character in the sequence.
     */
    public void changeCase () {
        int slength;                    /* the total length of the sequence */
        int writelen;                   /* the length of data to change case */
        String data = null;
        StringBuffer sequence = null;

        // copy - paste section
        if (!isSelectionEmpty()) {
            for (int y = minsy; y <= maxsy; y++) {
                sequence = datamodel.getLine(y).getSequence();
                slength = sequence.length();
                if (minsx < slength) {
                    writelen = (maxsx < slength ? maxsx : slength - 1);
                    data = sequence.substring(minsx, writelen);

                    if (Character.isUpperCase(data.charAt(minsx))) {
                        data.toLowerCase();
                    } else {
                        data.toUpperCase();
                    }
                    sequence.replace(minsx, writelen, data);
                }
            }
        }
    }

/////////////////////////////////////
//*********************************//
//* INTERNAL MANIPULATION METHODS *//
//*********************************//
/////////////////////////////////////
    /**
     * Inserts a string into the textarea's underlying Dataset
     * NOTE: this is a wrapper method for <b>insert</b>(<i>x</i>, <i>y</i>, <i>data</i>)
     ** 
     * @param x the X co-ordinate (column number) to insert the string at.
     * @param y the Y co-ordinate (row number) to insert the string at.
     * @param text the string to insert.
     */
    public void insert (final int x, final int y, char[] text) {
        Seq current = datamodel.getLine(y);
        int[] group = datamodel.getgroup(current.getGroupID());
        StringBuffer sequence = current.getSequence();

        if (group != null) {
            // itterate through the group and perform the mass insertion
            for (int number : group) {
                current = datamodel.getLine(number);
                sequence = current.getSequence();
                if (x < sequence.length() && !isProtectionsOn(current.getType(),
                        current.getProtectAlignment(), current.getProtectAmbiguous(),
                        current.getProtectUnambiguous(), text, 0, text.length)) {
                    sequence.insert(x, text);
                    current.modified();
                }
            }
        } else if (x < sequence.length() && !isProtectionsOn(current.getType(),
                current.getProtectAlignment(), current.getProtectAmbiguous(),
                current.getProtectUnambiguous(), text, 0, text.length)) {
            sequence.insert(x, text);
            current.modified();
        }
    }

    /**
     * Deletes characters from the textarea's underlying Dataset
     * NOTE: deletions are performed in a sequential manner
     **
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in sequences along the Y-axis).
     * @return a boolean corresponding to the result of the deletion (true = successful deletion)
     */
    public boolean delete (final int x, final int y, final int w, final int h) {
        boolean result = false;
        Seq current = null;
        int[] group;
        int max = y + h;

        for (int count = y; count >= max; count++) {
            current = datamodel.getLine(count);

            group = datamodel.getgroup(current.getGroupID());
            if (group != null) {
                // itterate through the group and perform the mass insertion
                for (int gln : group) {
                    if ((gln < y || gln > max) && gln < datamodel.getSize()) {
                        result |= pdelete(x, gln, w, 0);
                    } else if (gln >= datamodel.getSize()) {
                        BLMain.error("Invalid row number: " + gln, "boolean GDETextArea.insert (char)");
                    }
                }
            }
            result |= pdelete(x, count, w, 0);
        }
        return result;
    }

    /**
     * Removes text from one line of a document
     * (doesn't do any group processing - permission processing only).
     **
     * @param x the X-offset/column number to start the deletion from.
     * @param y the Y-offset/line number to delete characters from.
     * @param w the width of the deletion (measured in characters along the X-axis).
     * @param h the height of the deletion (measured in lines)
     * @return whether the deletion was a success.
     */
    public boolean pdelete (final int x, final int y, final int w, final int h) {
        char[] text = null;
        Seq current = null;
        int slength;
        int copylength;
        int xend = x + w;
        boolean result = (datamodel.getSize() == 0 && y == 0 && x == 0 && w == 0);		                        // the result of the operation (true if successful)
        StringBuffer sequence;

        // make sure all of the parameters are valid
        if (!result && datamodel.getSize() > 0
                && y >= 0 && y + h < datamodel.getSize() && x >= 0 && w > 0) {
            text = new char[w];
            for (int count = y; count <= y + h; count++) {
                current = datamodel.getLine(count);
                sequence = current.getSequence();
                slength = sequence.length();

                if (x + 1 < slength) {
                    copylength = (xend < slength ? xend : slength - 1);
                    sequence.getChars(x, copylength, text, 0);

                    if (!isProtectionsOn(current.getType(), current.getProtectAlignment(),
                            current.getProtectAmbiguous(), current.getProtectUnambiguous(),
                            text, 0, copylength + 1)) {
                        sequence.delete(x, x + w);
                        current.modified();
                        result = true;
                    }
                }
            }
        } else if (w == 0) {
            result = true;
        }


        return result;
    }

    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     **
     * @return whether or not the deletion was performed.
     */
    public boolean gdeleteSection () {
        final int x = minsx;
        final int y = minsy;
        final int w = maxsx - x;
        final int h = maxsy - y;
        boolean result = false;

        if (!isSelectionEmpty()) {
            if (delete(x, y, w, h)) {
                changePosition(false, x, y);
                result = true;
            }
        }
        return result;
    }

    /**
     * Used to delete the current selection before an insertion or as part of a deletion.
     * NOTE: non-grouped
     */
    public void deleteSelection () {
        final int x = minsx;
        final int y = minsy;
        final int w = maxsx - x;
        final int h = maxsy - y;

        if (!isSelectionEmpty() && pdelete(x, y, w, h)) {
            changePosition(false, x, y);
        }
    }

    /**
     * Retrieves the length of a line in the document.
     **
     * @param line the line line to find the length of.
     * @return the length of the line (in characters).
     */
    public int getLineLength (final int line) {
        Seq seq = datamodel.getLine(line);
        return (seq != null ? seq.getSequence().length() : 0);
    }

/////////////////////////////////
//*****************************//
//* PROTECTION STATUS METHODS *//
//*****************************//
/////////////////////////////////
    /**
     * Checks a string against all of a sequence's protection settings.
     *
     * This is done by obtaining the sequence's protection settings,
     * the type of the sequence, and whether the text contains a character in any
     * protected character class.
     **
     * @param type the type of the sequence to test against.
     * @param protect_align the status of alignment character protection
     * @param protect_ambig the status of ambiguous character protection
     * @param protect_uambig the status of unambiguous character protection
     * @param test the text to test.
     * @param start the start index of text to test
     * @param end the end index of text to test
     * @return true if the text violates the protection settings of the sequence.
     */
    public boolean isProtectionsOn (Seq.Type type,
            boolean protect_align, boolean protect_ambig, boolean protect_uambig,
            char[] test, int start, int end) {
        boolean protect = false;
        char current;

        if (protect_ambig || protect_uambig || protect_align) {
            if (type == Seq.Type.DNA || type == Seq.Type.RNA) {
                for (int count = start; !protect && count < end; count++) {
                    switch (Character.toLowerCase(test[count])) {
                        case 'b': case 'd': case 'h': case 'i': case 'k': case 'm':
                        case 'n': case 'r': case 's': case 'v': case 'w': case 'y':
                            protect = protect_ambig;
                            break;
                        case 'a': case 'c': case 'g': case 't': case 'u':
                            protect = protect_uambig;
                            break;
                        default:
                            protect = protect_align;
                            break;
                    }
                }
            } else if (type == Seq.Type.PROTEIN) {
                for (int count = start; !protect && count < end; count++) {
                    switch (Character.toLowerCase(test[count])) {
                        case ' ': case '\n': case '\t': case '\r': case '-':
                            protect = protect_align;
                            break;
                        case 'b': case 'x': case 'z': case '*':
                            protect = protect_ambig;
                            break;
                        default:
                            protect = protect_uambig;
                            break;
                    }
                }
            }
        }
        return protect;
    }

////////////////////////
//********************//
//* LISTENER METHODS *//
//********************//
////////////////////////
    /**
     * Called when a sequence is added to a Dataset.
     **
     * @param e the list data event corresponding to the addition
     */
    public void intervalAdded (ListDataEvent e) {
        updateLength(e.getIndex0(), e.getIndex1());     // update the size of the textarea

        final int x = 0;                                // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();       // the width of the area to repaint
        final int y = row2Y(e.getIndex0());             // the Y co-ordinate to start repainting at
        final int dy = row2Y(e.getIndex1() + 1);    	// the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight() - y;  // the height of the area to repaint

        // move the sequence down one notch
        this.getGraphics().copyArea(x, y, w, h, x, dy);
        repaint(x, y, w, dy - y);		// repaint the modified area of the textarea
    }

    /**
     * Called when a field in a sequence is modified.
     **
     * @param e the list data event corresponding to the modification
     */
    public void contentsChanged (ListDataEvent e) {
        updateLength(e.getIndex0(), e.getIndex1()); // update the size of the textarea

        final int x = 0;                            // the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();   // the width of the area to repaint
        final int y = row2Y(e.getIndex0());         // the Y co-ordinate to start repainting at
        final int h = row2Y(e.getIndex1() + 1);     // the height of the area to repaint

        repaint(x, y, w, h);                        // repaint the modified area of the textarea
    }

    /**
     * Called when a sequence is removed from a Dataset.
     **
     * @param e the list data event corresponding to the deletion
     */
    public void intervalRemoved (ListDataEvent e) {
        updateLength(e.getIndex0(), e.getIndex1());
        refreshSize();					// update the size of the textarea

        final int x = 0;				// the X co-ordinate to start repainting at
        final int w = (int) getSize().getWidth();	// the width of the area to repaint
        final int y = row2Y(e.getIndex1() + 1);			// the Y co-ordinate to start repainting at
        final int dy = row2Y(e.getIndex0());			// the Y co-ordinate to start repainting at
        final int h = (int) getSize().getHeight() - y;  // the height of the area to repaint

        this.getGraphics().copyArea(x, y, w, h, x, dy);
    }

/////////////////////////
//*********************//
//* SELECTION METHODS *//
//*********************//
/////////////////////////
    /**
     * Updates/moves the cursor to the new position.
     **
     * @param select whether or not the position should maintain selection status (i.e. true for SHIFT key).
     * @param newx the column co-ordinate of the new position.
     * @param newy the row co-ordinate of the new position.
     */
    protected void changePosition (boolean select, int newx, int newy) {
        final int maxlines = datamodel.getSize() - 1;
        final int oldx = this.column;
        final int oldy = this.row;

        // ensure that the new row and column are valid and
        // update the row and column
        this.row = Math.max(0, Math.min(newy, maxlines));
        this.column = Math.max(0, Math.min(newx, (getLineLength(newy))));

        if (select || oldx != sx || oldy != sy) {
            final int x = Math.min(oldx, minsx) * columnWidth;           // the X co-ordinate to start repainting at
            final int y = Math.min(oldy, minsy) * rowHeight;             // the Y co-ordinate to start repainting at
            final int w = (Math.max(oldx, maxsx) + 1) * columnWidth - x;	// the width of the area to repaint
            final int h = (Math.max(oldy, maxsy) + 1) * rowHeight - y;   // the height of the area to repaint

            if (!select) {
                sx = column;
                sy = row;
            }
            repaint(x, y, w, h);				// repaint the modified area of the textarea
        } else {
            sx = column;
            sy = row;

            // repaint the area of the old cursor
            // (automatically repaints the new cursor)
            repaint(oldx * columnWidth, oldy * rowHeight, columnWidth, rowHeight);
        }

        // update any cursor listeners
        canvas.cursorChange(column, row);

        scrollRectToVisible(new Rectangle(column * columnWidth, row * rowHeight, 1, 1));

        if (select && canvas != null) {
            canvas.selectionMade(this);
        }

        if (sx > column) {
            minsx = column;
            maxsx = sx;
        } else {
            minsx = sx;
            maxsx = column;
        }
        if (sy > row) {
            minsy = row;
            maxsy = sy;
        } else {
            minsy = sy;
            maxsy = row;
        }
        // TODO: remove
        repaint();
    }

    /**
     * Clears the current text selection
     */
    public final void clearSelection () {
        changePosition(false, column, row);
    }

    /**
     * Tests if the selection shape is empty
     *
     * @return true if the selection is empty
     */
    public final boolean isSelectionEmpty () {
        return (sx == column);
    }

////////////////////
//****************//
//* MATH METHODS *//
//****************//
////////////////////
    /**
     * Converts a row number into a Y co-ordinate
     **
     * @param r the row number
     * @return the corresponding Y value
     */
    protected final int row2Y (int r) {
        return (r * rowHeight);
    }

    /**
     * Returns the size of one row of the textarea.
     **
     * @return the row size of textarea.
     */
    public int rowSize () {
        return rowHeight;
    }

    /**
     * Returns the size of one column of the textarea.
     **
     * @return the column size of textarea.
     */
    public int columnSize () {
        return columnWidth;
    }

    /**
     * Updates the internal counter storing the length of the longest line
     **
     * @param start the first offset to examine for the update
     * @param end the last offset to examine for the update
     */
    public void updateLength (int start, int end) {
        int newlength;

        for (int count = start; count <= end; count++) {
            newlength = datamodel.getLine(count).getSequence().length() + 1;
            if (newlength > longestline) {
                longestline = newlength;
            }
        }
        refreshSize();
    }

    /**
     * Writes the selected sequences out to an appendable object
     **
     * @param format the file format to write
     * @param dest the destination of the sequence data
     * @throws IOException throws an IOException if there is any error appending the data
     */
    public void writeOut(DataFormat format, Appendable dest) throws IOException {
        final int length = maxsx - minsx;

        for (int lineNumber = minsy; lineNumber <= maxsy; lineNumber++) {
            format.translateTo(dest, datamodel.getLine(lineNumber), minsx, length);
        }
    }
}
