package org.biolegato.gdesupport.files;

import java.awt.Color;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import javax.swing.filechooser.FileFilter;
import org.biolegato.gdesupport.data.ColourMask;

/**
 * Used to select colour masks.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class CharacterColourMaskReader extends FileFilter {

    /**
     * Creates a new instance of GDEColourMaskFile
     */
    public CharacterColourMaskReader() {
    }

    /**
     * Determines whether a file is of type "character colour mask"
     *
     * @param f the file to test
     * @return true if the file is of type "character colour mask"
     */
    @Override
    public boolean accept(File f) {
        return true;
    }

    /**
     * Returns the description of "character colour mask"
     *
     * @return the string "Character-based colour mask file"
     */
    @Override
    public String getDescription() {
        return "Character-based colour mask file";
    }

    /**
     * Reads and parses a colour mask file.
     **
     * @param file the file to read.
     * @return an array of colour masks defined by the file.
     */
    public static ColourMask readColourMaskFile(File file) throws IOException{
        int tabIndex = 0;
        char[] characterArray = null;
        String line = "";
        String maskname = null;
        Color colour = Color.BLACK;
        BufferedReader reader = null;
        Color[] hash = new Color[ColourMask.HASH_SIZE];
        int hashpos;

        reader = new BufferedReader(new FileReader(file.getPath()));
        do {
            line = reader.readLine();
            if (line != null) {
                line = line.trim().toLowerCase();
                if ((!"aa\tcolour".equals(line) || !"aa\tcolor".equals(line))
                        && (tabIndex = line.indexOf('\t')) >= 0) {
                    characterArray = line.substring(0, tabIndex).trim().toCharArray();
                    colour = string2Colour(line.substring(tabIndex + 1).trim().toCharArray());

                    // TODO: add parsing work here
                    for (char character : characterArray) {
                        hashpos = ColourMask.hash(character);
                        if (hashpos >= 0) {
                            hash[hashpos] = colour;
                        }
                    }
                }
            }
        } while (line != null);
        maskname = file.getName().toLowerCase();
        if (maskname.endsWith(".csv")) {
            maskname = maskname.substring(0, maskname.length() - 4);
        }
        
        return new ColourMask(ColourMask.MaskType.CHARACTER, maskname, hash);
    }
    
    /**
     * Converts an array of characters to a colour object.
     *
     * @param arry the array of characters to parse.
     * @return the corresponding colour object.
     */
    private static Color string2Colour(char[] array) {
        int red;
        int green;
        int blue;
        int count = (array.length >= 1 && array[0] == '#' ? 1 : 0);
        Color result = Color.BLACK;

        // ensure there are enough charcters in the array to parse
        if (array.length - count >= 6) {
            red = hex2dec(array[count + 0]) * 16 + hex2dec(array[count + 1]);
            green = hex2dec(array[count + 2]) * 16 + hex2dec(array[count + 3]);
            blue = hex2dec(array[count + 4]) * 16 + hex2dec(array[count + 5]);
            result = new Color(red, green, blue);
        }

        return result;
    }

    /**
     * Converts a character from '0' - 'F' into a number from 0 - 16
     *
     * @param c the hex chracter to convert
     * @return the resulting decimal number
     */
    private static int hex2dec(char c) {
        int result = 0;

        if (c >= '1' && c <= '9') {
            result = 1 + (c - '1');
        } else if (c >= 'A' && c <= 'F') {
            result = 10 + (c - 'A');
        } else if (c >= 'a' && c <= 'f') {
            result = 10 + (c - 'a');
        }
        return result;
    }
}
