package org.biolegato.gdesupport.files;
/*
 * GDEFile.java
 *
 * Created on January 30, 2008, 11:58 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import org.biolegato.gdesupport.data.Seq;
import org.biolegato.gdesupport.data.Seq.Direction;
import org.biolegato.gdesupport.data.Seq.Strandedness;
import org.biolegato.gdesupport.data.Seq.Topology;
import org.biolegato.gdesupport.data.Seq.Type;
import org.biolegato.main.BLMain;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.biolegato.gdesupport.data.Dataset;

/**
 * This class acts as a parser/translator for standard GDE files.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEFile extends DataFormat {

    /**
     * Creates a new instance of GDEFile
     */
    public GDEFile () {
    }

    /**
     * Translates a sequence into the GDE file format.
     *
     * @param seq the sequence to translate
     * @return the resulting string
     */
    public void translateTo (Appendable result, Seq seq, int offset, int length) throws IOException {
        // translate the data
        // NOTE: append is faster than + or concat operators
        if (seq != null) {
            result.append("{\nname            \"").append(
		    quote(seq.getName())).append(
// TODO: reimplement longname
//			"\"\nlongname        \"").append(quote(seq.getName())).append(
// TODO: reimplement accession numbers
//			"\"\nsequence-ID     \"").append(quote(seq.getAccession())).append(
			"\"\ncreation-date   \"").append((new SimpleDateFormat("MM/dd/yy kk:mm:ss")).format(new Date())).append(
			"\"\ndirection       ").append((Direction.FROM5TO3.equals(seq.getDirection()) ? "1" : "-1")).append(
			"\nstrandedness    ").append((Strandedness.SINGLE.equals(seq.getStrandedness()) ? "1" : "2")).append(
			"\ntype            ").append(blTypetoGDE(seq.getType())).append(
			"\n").append((Topology.CIRCULAR.equals(seq.getTopology()) ? "circular        1\n" : "")).append(
			"offset          0\ngroupID         ").append(Integer.toString(Math.min(0, seq.getGroupID() - 1))).append(
			"\ncreator         \"").append(quote("")).append(
			"\"\ndescrip         \"").append(quote(seq.getDescription())).append(
// TODO: reimplement comments section?
/*			"\"\ncomments        \"").append(quote(seq.get("comments"))).append(*/
			"\"\nsequence        \"").append(quote(seq.getSequence().substring(offset, length))).append("\"\n}\n");
        }
    }

    /**
     * Translates data in the GDE file format to sequence objecte
     *
     * @param data the buffered reader to parse
     * @return the translated sequences
     * @throws IOException any exeptions that occur while reading the stream are passed
     */
    public void translateFrom (Dataset datamodel, java.io.BufferedReader data) throws
            IOException {
        int y = datamodel.getSize();
        int groupID = 0;
        Type type = Type.DNA;
        String name = "";
        String sequence = "";
        Direction direction = Direction.FROM5TO3;
        Strandedness strandedness = Strandedness.SINGLE;
        Topology topology = Topology.LINEAR;
        String description = null;
        String line = "";
        String fname = "";
        String fvalue = "";
        StringBuffer fvalueBuffer;                                              // used for reading multiple lines

        // translate the data
        while ((line = data.readLine()) != null) {
            line = line.trim();
            if ( ! "".equals(line)) {
                if (line.contains("}")) {
                    datamodel.addSequence(y, new Seq(type, name, new StringBuffer(sequence),
                            direction, topology, strandedness, groupID - 1, description));
                    y++;
                    line = line.substring(line.indexOf('{') + 1).trim();

                    // reset all values for the next loop itteration
                    groupID = 0;
                    type = Type.DNA;
                    name = "";
                    sequence = "";
                    direction = Direction.FROM5TO3;
                    strandedness = Strandedness.SINGLE;
                    topology = Topology.LINEAR;
                    description = null;
                }
                if (line.contains(" ") && ( ! line.contains("\t") ||
                                           line.indexOf(' ') <
                                           line.indexOf('\t'))) {
                    fname = line.substring(0, line.indexOf(' ')).trim();
                    fvalue = line.substring(line.indexOf(' ') + 1).trim();
                } else if (line.contains("\t")) {
                    fname = line.substring(0, line.indexOf('\t')).trim();
                    fvalue = line.substring(line.indexOf('\t') + 1).trim();
                } else {
                    fname = "";
                    fvalue = "";
                }
                if (fvalue.startsWith("\"")) {
                    fvalueBuffer = new StringBuffer();
                    fvalue = fvalue.substring(1);
                    fvalueBuffer.append(fvalue);
                    while (fvalue != null && ! fvalue.contains("\"")) {
                        // NOTE: append is faster than + or concat operators
                        fvalue = data.readLine();
                        fvalueBuffer.append(fvalue);
                    }
                    fvalue = fvalueBuffer.toString();
                    fvalue = fvalue.substring(0, fvalue.indexOf('\"'));
                }
                if ("".equals(fname) && "".equals(fvalue)) {
                } else if ("type".equals(fname)) {
                    if ("RNA".equalsIgnoreCase(fvalue)) {
                        type = Type.RNA;
                    } else if ("PROTEIN".equalsIgnoreCase(fvalue)) {
                        type = Type.PROTEIN;
                    } else if ("TEXT".equalsIgnoreCase(fvalue)) {
                        type = Type.TEXT;
                    } else if ("MASK".equalsIgnoreCase(fvalue)) {
                        type = Type.MASK;
                    } else {
                        type = Type.DNA;
                    }
                } else if ("groupID".equals(fname)) {
		    if (BLMain.testNumber(fvalue.trim())) {
			try {
			    groupID = Integer.parseInt(fvalue);
			} catch (Exception ex) {
			}
		    }
                } else if ("circular".equals(fname)) {
                    if ("0".equals(fvalue)) {
                        topology = Topology.LINEAR;
                    } else {
                        topology = Topology.CIRCULAR;
                    }
                } else if ("direction".equals(fname)) {
                    if ("-1".equals(fvalue)) {
                        direction = Direction.FROM3TO5;
                    } else {
                        direction = Direction.FROM5TO3;
                    }
                } else if ("strandedness".equals(fname)) {
                    if ("1".equals(fvalue)) {
                        strandedness = Strandedness.SINGLE;
                    } else {
                        strandedness = Strandedness.DOUBLE;
                    }
// TODO: reimplement cration date for GDE files
/*              } else if ("creation-date".equals(fname)) {
                    creationDate = new Date(fvalue);*/
// TODO: reimplement creator section for GDE files
/*                } else if ("creator".equals(fname)) {
                    sequence.put("creator", fvalue);*/
// TODO: reimplement accession numbers
/*                } else if ("sequence-ID".equals(fname)) {
                    sequence.setAccession(value);*/
                } else if ("name".equals(fname)) {
                    name = fvalue;
  // TODO: reimplement longname section for GDE files
/*              } else if ("longname".equals(fname)) {
                    sequence.put("longname", fvalue);*/
                } else if ("descrip".equals(fname)) {
                    description = fvalue;
// TODO: reimplement comments section for GDE files
/*                } else if ("comments".equals(fname)) {
                    sequence.put("comments", fvalue);*/
                } else if ("sequence".equals(fname)) {
                    sequence = fvalue;
                } else {
                    BLMain.warning("Unsupported attribute (" +
                                                           fname + " = " +
                                                           fvalue + " )",
                                                           "GDEFile plugin");
                }
            }
        }
        if ( name != null ) {
            datamodel.addSequence(y, new Seq(type, name, new StringBuffer(sequence)));
            y++;
        }
    }

    /**
     * Determines whether or not a specified file is of type GDE file (based on extension)
     * Currently the only extension supported is ".gde".
     *
     * @param file the file to test
     * @return true if the file is of type GDE file (otherwise false)
     * @see javax.swing.filechooser.FileFilter#accept
     */
    public boolean accept (File file) {
        return (file.isDirectory() || file.getAbsolutePath().toLowerCase().
                endsWith(".gde"));
    }

    /**
     * Returns a description of the file format that can be displayed to the user.
     *
     * @return the string description of the file format
     * @see javax.swing.filechooser.FileFilter#getDescription
     */
    public String getDescription () {
        return "GDE file (*.gde)";
    }

    /**
     * Quotes a GDE string (replaces " with ' and {} with [])
     * 
     * @param input	the string to quote
     * @return the quoted string
     */
    public static String quote (String input) {
        return input.replaceAll("\"", "\'").replaceAll("\\{", "\\[").replaceAll(
                "\\}", "\\]");
    }

    /**
     * Converts a sequence's type enum to the GDE file format
     *
     * @param type the type enum to convert
     * @return the GDE equivilent
     */
    private static String blTypetoGDE (Type type) {
        String result = "DNA";
        switch (type) {
            case RNA:
                result = "RNA";
                break;
            case PROTEIN:
                result = "PROTEIN";
                break;
            case MASK:
                result = "MASK";
                break;
            case TEXT:
                result = "TEXT";
                break;
        }
        return result;
    }

    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the reader to parse data from
     * @return whether or not the format is correct
     */
    @Override
    public boolean isFormat (Reader test) {
        int check = ' ';
        try {
            while (check == ' ' || check == '\t' || check == '\n' || check ==
                                                                     '\r') {
                test.mark(2);
                check = test.read();
            }
            test.reset();
        } catch (Throwable e) {
            e.printStackTrace(System.err);
        }
        return (check == '{');
    }

}
