/*
 * Dataset.java
 *
 * Created on September 30, 2008, 11:08 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.data;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.swing.AbstractListModel;

/**
 * The internal document format for BioLegato.
 * <p>
 * This document is structured as a linked list of sequences.  Each character has an offset based on its
 * position within the list and it's position within its containing sequence.  Sequences highindex at 0
 * (first character in the first sequence in the list, and lowindex with the last character in the last sequence
 * within the list.
 * </p>
 **
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class Dataset extends AbstractListModel {

    /**
     * This linked list used to store all of the height in the document.
     * <p>
     *  Each y is stored as a linked list of sequence wrappers.
     *  Each sequence wrapper is characterized by a sequence and a highindex offset.
     *  The highindex offset is used to determine the offset of the sequence within
     *  the document.
     * </p>
     *
     * @see org.biolegato.core.data.seqdoc.SeqWrap
     */
    private final List<Seq> lines = new ArrayList<Seq>();
    /**
     * Keeps track of the maximum groupID number in usage.
     */
    private int maxGroup = 0;
    /**
     * A hashtable used to map all groups to sequences 
     */
    private Set<Seq>[] group2seq = new Set[DEFAULT_MAX_GROUPS];
    /**
     * The maximum number of groups
     */
    private final static int DEFAULT_MAX_GROUPS = 30;

    /**
     * Creates a new instance of Dataset
     */
    public Dataset() {
    }

////////////////////////////////
//****************************//
//* DIRECT DATA MODIFICATION *//
//****************************//
////////////////////////////////
    /**
     * Adds a sequence to the data container.  This function calls all listeners.
     **
     * @param y is the y number to insert the sequence.
     * @param seq is the sequence to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean addSequence(int y, Seq seq) {
        boolean result = false;

        // TODO: add x support
        if (seq != null && y >= 0 && y <= getSize()) {
            lines.add(y, seq);
            fireIntervalAdded(this, y, y);
            result = true;
        }
        return result;
    }

    /**
     * Adds sequences to the data container.  This function calls all listeners.
     **
     * @param y is the y number to insert the sequence.
     * @param seqs are the sequences to insert.
     * @return true if the insertion was successful, otherwise false.
     */
    public boolean addSequences(int y, Collection<Seq> seqs) {
        boolean result = false;

        // TODO: add x support
        if (seqs != null && seqs.size() > 0 && y >= 0 && y <= getSize()) {
            lines.addAll(y, seqs);
            fireIntervalAdded(this, y, y + seqs.size() - 1);
            result = true;
        }
        return result;
    }

    /**
     * Removes an array of sequences from the data container.
     **
     * @param lineNumbers the y numbers to remove.
     */
    public void removeSequences(final int[] lineNumbers) {
        // 'highindex' corresponds to the largest number index for the interval
        // we are currently exploring within the lineNumbers array
        int highindex = lineNumbers.length - 1;

        // 'lowindex' corresponds to the smallest number index for the interval
        // we are currently exploring within the lineNumbers array
        int lowindex = highindex;

        // sort the y numbers
        // explanation of necessity to follow
        Arrays.sort(lineNumbers);

        // ensure that there are height to delete
        if (!lines.isEmpty() && lineNumbers != null) {
            /**
             * itterate backwards through each sequence y number and delete it
             * this ensures that when deleting a line number, other line number
             * indices will not change when deleting
             *
             * should other numbers be changed, we would accidentally delete
             * the wrong sequence
             ****
             * For example:
             ****
             *  suppose we have the following data
             *   [0] = 'abcdef'
             *   [1] = 'ghij'
             *   [2] = 'klmn'
             *   [3] = 'opqr'
             *
             *  suppose we call removeSequences (new int[] {1,2,0});
             *
             *  the result we would expect would be
             *   [0] = 'abcdef'
             *
             *  if we were not to change sort the line numbers and itterate backwards
             *  we would first delete sequence 1:
             *   [0] = 'abcdef'
             *   [1] = 'klmn'
             *   [2] = 'opqr'
             *  as you can see the indices changed, therefore when we delete sequence 2, we would get
             *   [0] = 'abcdef'
             *   [1] = 'klmn'
             *  and finally
             *   [0] = 'klmn'
             *
             *  NOTE: this is not our expected result!
             *
             * This can only be solved by sorting the sequence numbers, and removing
             * them from highest to lowest (i.e. sorted array parsed in reversed order)
             *
             * The reason we will not reverse the array, is only efficiency
             * and to not reinvent the wheel (Arrays.sort is built in, and reversal
             * methods would take more time and memory)
             **/
            while (lowindex >= 0) {
                // make highindex and lowindex the same value
                highindex = lowindex;

                /**
                 * try to form an interval
                 **
                 * the reason why we do this: because we want to minimize the number of calls
                 * to the listener methods (i.e. increase the speed of the program)
                 *
                 * the reason this optimization is necessary, is because creating objects
                 * is very expensive, and each time we call ListDataListener.intervalRemoved,
                 * we must create a new ListDataEvent object.
                 */
                while (lowindex > 0 && lineNumbers[lowindex - 1] == lineNumbers[lowindex] - 1) {
                    ungroup(lines.remove(lineNumbers[lowindex]));
                    lowindex--;
                }

                // remove the final line (NOTE: the above method will remove every
                // sequence in the interval, EXCEPT the last one (i.e. the lowest index
                ungroup(lines.remove(lineNumbers[lowindex]));

                // send an interval removed event to all of this dataset's ListDataListeners
                fireIntervalRemoved(this, lineNumbers[lowindex], lineNumbers[highindex]);
                lowindex--;
            }
        }
    }

//////////////////////
//******************//
//* DATA RETRIEVAL *//
//******************//
//////////////////////

    /**
     * Returns the number of lines in the document.
     **
     * @return the number of lines in the document.
     */
    public int getSize() {
        return lines.size();
    }
    
    /**
     * Retrieves a sequence object from the Dataset specified by its y y.
     * 
     * @param number the y y to retrieve the sequence.
     * @return the sequence.
     */
    public Seq getLine(final int number) {
        return (number >= 0 && number < getSize() ? lines.get(number) : null);
    }

    /**
     * Called when a field in a sequence is modified.
     **
     * @param sequence the sequence modified.
     * @param key the key of the modified field in the sequence.
     */
    void sequenceChanged(final int index) {
        fireContentsChanged(this, index, index);
    }

///////////////
//***********//
//* GENERAL *//
//***********//
///////////////

    public void group(int[] sequences) {
        Seq seq;
        int maxGroupStart = maxGroup;

        // ensure that there is an entry in group2seq to add the sequence to.
        while (group2seq[maxGroup] != null) {
            maxGroup++;
            maxGroup %= (group2seq.length - 1); // ensures wraparound
            
            /*
             * Should virtually never be run (except if someone is using a lot of groups)
             */
            if (maxGroup == maxGroupStart) {
                Set<Seq>[] temp = new Set[group2seq.length + DEFAULT_MAX_GROUPS];
                System.arraycopy(group2seq, 0, temp, 0, group2seq.length);
                group2seq = temp;
                break;
            }
        }
        group2seq[maxGroup] = new HashSet<Seq>();

        // TODO: Collections.sort();
        for (int y : sequences) {
            seq = lines.get(y);
            ungroup(seq);
            seq.groupID = maxGroup + 1;
            sequenceChanged(lines.indexOf(seq));
            group2seq[maxGroup].add(seq);
        }
    }

    public void ungroup(int[] sequences) {
        Seq seq;
        for (int y : sequences) {
            seq = lines.get(y);
            ungroup(seq);
            seq.groupID = 0;
            sequenceChanged(lines.indexOf(seq));
        }
    }

    private void ungroup(Seq seq) {
        Set<Seq> group = null;

        if (seq != null) {
            // remove previous groupID reference
            if (seq.groupID > 0 && seq.groupID <= group2seq.length) {
                group = group2seq[seq.groupID - 1];
                if (group != null) {
                    group.remove(seq);
                    if (group.size() <= 0) {
                        group2seq[seq.groupID - 1] = null;
                    }
                }
            }
        }
    }

    public int[] getgroup(int groupNumber) {
        int count = 0;
        int[] result = null;
        Set<Seq> group;

        if (groupNumber > 0 && groupNumber <= group2seq.length && group2seq[groupNumber - 1] != null) {
            group = group2seq[groupNumber - 1];
            result = new int[group.size()];
            for (Seq c : group) {
                System.out.println(" indexof: " + c.getName() + "  " + lines.indexOf(c));
                result[count] = lines.indexOf(c);
                count++;
            }
        } else {
            System.out.println("INVALID!");
        }
        
        return result;
    }

////////////////////////////
//************************//
//* LIST MODEL FUNCTIONS *//
//************************//
////////////////////////////
    /**
     * Returns the names of sequences based on their y number.
     *
     * @param index the y number to obtain the name of.
     * @return returns the name of sequence in the sequence document at the y number indicated by index.
     */
    public Object getElementAt(int index) {
        StringBuilder name = new StringBuilder();
        Seq seq = getLine(index);

        if (seq != null) {
            if (seq.groupID > 0) {
                name.append(seq.groupID).append(" ");
            } else {
                name.append("_ ");
            }
            name.append(seq.getName());
        }
        return name;
    }
}
