package org.biolegato.gdesupport.files;

import java.awt.Color;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import javax.swing.filechooser.FileFilter;
import org.biolegato.gdesupport.data.ColourMask;
import org.biolegato.main.BLMain;

/**
 * Used to select colour masks.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public final class GDEColourMaskFile extends FileFilter {

    private static final Color[] stdcolours = new Color[]{
        /*
         * DEFAULT GDE COLOURS
         */
        // Red
        new Color(255, 0, 0),
        // Orange
        new Color(240, 180, 20),
        // Blue
        new Color(0, 0, 255),
        // Black
        new Color(0, 0, 0),
        // Green
        new Color(0, 200, 0),
        // Pink
        new Color(255, 0, 160),
        // Tirquoise
        new Color(0, 160, 200),
        // Cyan
        new Color(0, 127, 255),
        // Olive 2
        new Color(105, 139, 30),
        // Purple
        new Color(128, 0, 255),
        // Blue-grey
        new Color(125, 158, 192),
        // Dark yellow
        new Color(205, 205, 0),
        // Deep pink
        new Color(139, 10, 80),
        // Burnt sienna
        new Color(233, 116, 81),
        // Medium tirquoise
        new Color(0, 180, 127),
        // Charteuse (med)
        new Color(127, 230, 0)};

    /**
     * Creates a new instance of GDEColourMaskFile
     */
    public GDEColourMaskFile () {
    }

    @Override
    public boolean accept (File f) {
        return true;
    }

    @Override
    public String getDescription () {
        return "GDE colour mask file";
    }

    /**
     * Reads and parses a colour mask file.
     *
     * @param file the file to read.
     * @return an array of colour masks defined by the file.
     */
    public static ColourMask readColourMaskFile (File file) throws IOException {
        int number = 1;
        int length = 0;
        String line = "";
        String name = file.getName();
        int colourNumber = 0;
        Color[] colours = null;
//        boolean nodash = false;
        BufferedReader reader = null;

        reader = new BufferedReader(new FileReader(file.getPath()));
        line = reader.readLine();
        if (line != null && line.toLowerCase().startsWith("name:")) {
            name = line.substring(5);
            line = reader.readLine();
        }
        if (line != null && line.toLowerCase().startsWith("length:")) {
            try {
                length = Integer.parseInt(line.substring(7));
                colours = new Color[length];
            } catch (Throwable nfe) {
                nfe.printStackTrace(System.err);
            }
            line = reader.readLine();
        } else {
            BLMain.error("Invalid colour GDE mask file - missing \"length:\" field!", "GDEColourMaskColourMask");
        }
        if (line != null && line.toLowerCase().startsWith("nodash:")) {
            BLMain.warning("Biolegato does not support \"nodash:\" in GDE colour mask files", "GDEColourMaskColourMask");
//                    nodash = true;
            line = reader.readLine();
        }
        if (line != null && line.toLowerCase().startsWith("start:")) {
            line = reader.readLine();
            for (number = 0; line != null && number < length; number++) {
                line = line.trim();
                if (BLMain.testNumber(line)) {
                    try {
                        colourNumber = Integer.parseInt(line);
                        if (colourNumber > 0 && colourNumber <= stdcolours.length) {
                            colours[number] = stdcolours[colourNumber - 1];
                        } else {
                            BLMain.error("Invalid colour #" + colourNumber, "GDEColourMaskColourMask");
                        }
                    } catch (NumberFormatException nfe) {
                        nfe.printStackTrace(System.err);
                        colourNumber = -1;
                    }
                } else {
                    BLMain.error("\"" + line + "\" is not a colour", "GDEColourMaskColourMask");
                }
                line = reader.readLine();
            }

            if (number < length) {
                Color[] temp = colours;
                colours = new Color[number];
                System.arraycopy(temp, 0, colours, 0, colours.length);
                BLMain.error("Reached end of file before end of colour", "GDEColourMaskColourMask");
            }
        } else {
            BLMain.error("Missing GDE colour mask data!", "GDEColourMaskColourMask");
        }

        return new ColourMask(ColourMask.MaskType.POSITION, name, colours);
    }

    /**
     * Converts a string to an integer (note: returns 0 on failure).
     *
     * @param s the string to convert.
     * @return the integer representation of the string, 0 on failure.
     */
    public static int stringToInt (String s) {
        int i = 0;
        if (BLMain.testNumber(s)) {
            try {
                i = Integer.parseInt(s);
            } catch (NumberFormatException nfe) {
            }
        }
        return i;
    }
}
