/*
 * GDEMenuParser.java
 *
 * Created on June 9, 2010, 1:28 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.menu;

import java.awt.event.ActionListener;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.JMenuItem;
import org.biolegato.main.BLMain;

/**
 *
 * @author alvare
 */
public class GDEMenuParser {

    private static final String[] BLANK_STRING = new String[0];

    public enum WidgetType {

        CHOICE_LIST,
        CHOOSER,
        COMBOBOX,
        FILE,
        TEXT,
        SLIDER;

        public static WidgetType toType(String test) {
            int length = test.length();     // obtain the length for comparison
            WidgetType type = TEXT;         // default to text field

            if (length == 6 && "slider".equalsIgnoreCase(test)) {
                type = SLIDER;
            } else if (length == 7 && "chooser".equalsIgnoreCase(test)) {
                type = CHOOSER;
            } else if (length == 11) {
                if ("choice_menu".equalsIgnoreCase(test)) {
                    type = COMBOBOX;
                } else if ("choice_list".equalsIgnoreCase(test)) {
                    type = CHOICE_LIST;
                }
            } else if (length == 12 && "file_chooser".equalsIgnoreCase(test)) {
                type = FILE;
            }

            return type;
        }
    }

    /**
     * Reads in the default .GDEMenus files.
     */
    public static void loadGDEMenus() {
        try {
            // check the settings before reading in the menus
            if (!"true".equalsIgnoreCase(BLMain.getProperty("GDE.menu.disable"))) {
                // read in menus from the GDE directory
                if (System.getenv("GDE_HELP_DIR") != null) {
                    GDEMenuParser.readGDEMenuFile(new File(System.getenv("GDE_HELP_DIR") + File.separator + ".GDEmenus"));
                } else {
                    BLMain.warning("GDE_HELP_DIR not set!", "GDEMenu plugin");
                }

                // read in menus from the current working directory
                GDEMenuParser.readGDEMenuFile(new File(BLMain.PROGRAM_DIR + File.separator + ".GDEmenus"));

                // read in menus from the home directory
                if (BLMain.HOME_DIR != null) {
                    GDEMenuParser.readGDEMenuFile(new File(BLMain.HOME_DIR + File.separator + ".GDEmenus"));
                } else {
                    BLMain.warning("HOME not set!", "GDEMenu plugin");
                }
            }
        } catch (Throwable e) {
            e.printStackTrace(System.err);
        }
    }

    /**
     * Reads in the makemenus GDE menus.
     */
    public static void loadMakemenus() {
        String line;
        File menudirfile;
        String basedir = System.getenv("GDE_MAKEMENUS_DIR");
        File menulist = null;
        Map<String, Set<String>> directoryList = new HashMap<String, Set<String>>();
        BufferedReader localreader = null;				    // the BufferedReader for reading the GDE menu

        // read in menus from the GDE directory
        if (basedir != null) {
            menudirfile = new File(basedir + File.separator + "ldir.param");
            if (menudirfile.exists() && menudirfile.canRead() && menudirfile.isFile()) {
                try {
                    localreader = new BufferedReader(new FileReader(menudirfile));
                    while ((line = localreader.readLine()) != null) {
                        readMenulistFile (directoryList, line);
                    }
                    localreader.close();
                    localreader = null;
                } catch (IOException ioe) {
                    ioe.printStackTrace(System.err);
                }
            }
            try {
                readMenulistFile (directoryList, basedir + File.separator + "menus");
            } catch (IOException ioe) {
                ioe.printStackTrace(System.err);
            }
        } else {
            BLMain.warning("GDE_MAKEMENUS_DIR not set!", "GDE Makemenus plugin");
        }
    }

    private static void readMenulistFile (Map<String, Set<String>> directoryList, String basename) throws IOException {
        int tabindex;
        boolean possible;
        char[] system;
        String line;
        String dirname = null;
        File menulist = new File(basename + File.separator + "menulist");
        File menufile = null;
        BufferedReader reader = null;					    // the BufferedReader for reading the GDE menu

        if (menulist.exists() && menulist.canRead() && menulist.isFile()) {
            reader = new BufferedReader(new FileReader(menulist));

            while ((line = reader.readLine()) != null) {
                possible = true;
                if (line.length() > 0 && !line.trim().equals("") && !line.startsWith("#")) {
                    if (Character.isWhitespace(line.charAt(0))) {
                        if (dirname != null && !directoryList.get(dirname).contains(line)) {
                            // menu item
                            line = line.trim();
                            tabindex = line.indexOf('\t');
                            if (tabindex >= 0) {
                                // TODO system specific!
                                system = line.substring(tabindex + 1).trim().toCharArray();
                                line = line.substring(0, tabindex);
                                if (system.length > 0) {
                                    possible = false;
                                    for (char av : system) {
                                        possible = possible || (av == 'S' && "solaris-sparc".equalsIgnoreCase(System.getenv("BIRCH_PLATFORM")))
                                                || (av == 's' && "solaris-amd64".equalsIgnoreCase(System.getenv("BIRCH_PLATFORM")))
                                                || (av == 'L' && "linux-intel".equalsIgnoreCase(System.getenv("BIRCH_PLATFORM")))
                                                || (av == 'l' && "linux-x86_64".equalsIgnoreCase(System.getenv("BIRCH_PLATFORM")));
                                    }
                                    if (BLMain.debug) {
                                        BLMain.message(line + "    " + String.valueOf(system) + "    " + System.getenv("BIRCH_PLATFORM") + "    " + (possible ? "loaded" : "NOT loaded"), "BIRCH Makemenus");
                                    }
                                }
                            }

                            if (possible) {
                                menufile = new File(basename + File.separatorChar + dirname + File.separatorChar + line + ".item");
                                if (!menufile.exists()) {
                                    BLMain.error("Cannot read menu item - file does not exist", "GDEMakeMenus plugin");
                                } else if (!menufile.isFile()) {
                                    BLMain.error("Cannot read menu item - location does not point to a file", "GDEMakeMenus plugin");
                                } else if (!menufile.canRead()) {
                                    BLMain.error("Cannot read menu item - permission denied", "GDEMakeMenus plugin");
                                } else {
                                    // everything is OK, read the menu file
                                    GDEMenuParser.readGDEMenuFile(dirname, menufile);
                                }
                            }
                        }
                    } else {
                        // dirname
                        dirname = line;
                        if (!directoryList.containsKey(dirname)) {
                            directoryList.put(dirname, new HashSet<String>());
                        }
                    }
                }
            }
            reader.close();
        }
    }

    /**
     * This function loads a GDE formatted menu file into the program.
     **
     * @param file the file to read.
     */
    public static void readGDEMenuFile(File file) {
        readGDEMenuFile("File", file);
    }

    /**
     * This function loads a GDE formatted menu file into the program.
     **
     * @param menuName the name of the current heading.
     * @param file the file to read.
     */
    public static void readGDEMenuFile(String menuName, File file) {
        int tmpindex = 0;
        String line = "";                                                           // the current line in the file
        String fname = "";                                                          // The current field name being parsed
        String fvalue = "";                                                         // The current field value
        BufferedReader reader = null;                                               // the BufferedReader for reading the GDE menu
        Set<String> argList = new HashSet<String>();                                // used for replacing GDE regular variable names with BioLegato variable names
        List<String> fileVariables = new LinkedList<String>();                      // used for replacing GDE file variable names with BioLegato variable names
        List<String> choicenames = new LinkedList<String>();                        // Used for storing variable choice names
        List<String> choicevalues = new LinkedList<String>();                       // Used for storing variable choice values
        Map<String, Widget> widgets = new LinkedHashMap<String, Widget>();
        boolean saveMenuItem;

        String itemName = null;                                                       // The current menu item's name
        String command = null;
        ActionListener itemAction = null;
        JMenuItem jmiresult = null;                                                 // temporary vairble for adding action listeners to the JMenuItem
        String runCommand = null;                                                   // The data for the current run button (used only on button creation)
        String helpCommand = null;                                                  // The data for the current help button (used only on button creation)

        WidgetType argtype = null;
        String argname = null;
        String arglabel = null;
        int argintvalue = 0;
        int minimum = 0;
        int maximum = 500000;
        String argdefault = null;

        String inname = null;
        String informat = null;
        // Determines whether or not to delete the file after execution.
        boolean insave = false;

        String outname = null;
        String outformat = null;
        // Determines whether or not to overwrite the file if it already exists.
        boolean outsave = false;
        // Determines whether or not to overwrite the file if it already exists.
        boolean outoverwrite = false;

        if (file.exists() && file.isFile() && file.canRead()) {
            try {
                // open the file
                reader = new BufferedReader(new FileReader(file));
                do {
                    // read the next line
                    fname = "";
                    fvalue = "";
                    line = reader.readLine();

                    // skip comments and blank lines and make sure that there is a :
                    if (line != null && !"".equals(line) && (tmpindex = line.indexOf(':')) >= 0 && (!line.startsWith("#") || line.startsWith("#@"))) {
                        // remove biolegato specific parsing header (since we wish to parse these commands)
                        if (line.startsWith("#@")) {
                            line = line.substring(2);
                        }

                        // the first parameter is the field name
                        // the second parameter is the field data
                        fname = line.substring(0, tmpindex).trim().toLowerCase();
                        fvalue = line.substring(tmpindex + 1).trim();
                    }

                    // cache the value of this for faster if statements
                    saveMenuItem = line == null || "item".equals(fname) || "menu".equals(fname);

//////////////////////////////////////////////////
//**********************************************//
//* add any pending arguments to the menu item *//
//**********************************************//
//////////////////////////////////////////////////
                    if (itemName != null) {
                        if (saveMenuItem || "in".equals(fname)) {
                            if (inname != null && informat != null) {
                                widgets.put(inname, new TempFile(true, false, insave, false, informat));
                                fileVariables.add(inname.toLowerCase());
                                informat = null;
                                insave = false;
                            }

                            ///// PERFORMANCE /////
                            // *** DO INSTEAD OF "in".equals(fvalue) ***//
                            if (!saveMenuItem) {
                                // create a new argument parameter
                                inname = fvalue.toLowerCase();
                            } else {
                                inname = null;
                            }
                        }

                        if (saveMenuItem || "out".equals(fname)) {
                            if (outname != null && outformat != null) {
                                widgets.put(outname, new TempFile(false, true, outsave, outoverwrite, outformat));
                                fileVariables.add(outname.toLowerCase());
                                outformat = null;
                                outoverwrite = false;
                            }

                            ///// PERFORMANCE /////
                            // *** DO INSTEAD OF "out".equals(fvalue) ***//
                            if (!saveMenuItem) {
                                // create a new argument parameter
                                outname = fvalue.toLowerCase();
                            } else {
                                outname = null;
                            }
                        }

                        if ((saveMenuItem || "arg".equals(fname))) {
                            if (argname != null) {
                                // this is necessary before doing any other processing
                                argList.add(argname.toLowerCase());

                                // PERFORM SHARED CODE
                                if (argtype == WidgetType.CHOICE_LIST || argtype == WidgetType.CHOOSER || argtype == WidgetType.COMBOBOX || argtype == WidgetType.SLIDER) {
                                    if (argdefault != null) {
                                        try {
                                            argintvalue = Integer.parseInt(argdefault);
                                        } catch (NumberFormatException nfe) {
                                        }
                                    }
                                    // BRANCH BETWEEN SLIDER AND LIST CODE!!!
                                    if (argtype == WidgetType.SLIDER) {
                                        minimum = Math.max(0, minimum);
                                        maximum = Math.max(minimum, maximum);
                                        argintvalue = Math.max(Math.min(argintvalue, maximum), minimum);
                                        widgets.put(argname, new NumberWidget(arglabel, minimum, maximum, argintvalue));
                                    } else {
                                        String[] choices = choicevalues.toArray(BLANK_STRING);
                                        String[] choicenamearray = choicenames.toArray(BLANK_STRING);
                                        for (int count = 0; count < choices.length; count++) {
                                            choices[count] = GDE2BLArgs(argList, fileVariables, choices[count]);
                                        }
                                        argintvalue = Math.max(Math.min(argintvalue, choices.length - 1), 0);
                                        switch (argtype) {
                                            case CHOICE_LIST:
                                                BLMain.warning("Using deprecated menu widget \"choice_list\"", "GDE menu parser");
                                                widgets.put(argname, new ChoiceList(arglabel, choicenamearray, choices, argintvalue));
                                                break;
                                            case CHOOSER:
                                                widgets.put(argname, new Chooser(arglabel, choicenamearray, choices, argintvalue));
                                                break;
                                            case COMBOBOX:
                                                widgets.put(argname, new ComboBoxWidget(arglabel, choicenamearray, choices, argintvalue));
                                                break;
                                        }
                                    }
                                } else if (argtype == WidgetType.FILE) {
                                    widgets.put(argname, new FileChooser(arglabel, argdefault));
                                } else {
                                    widgets.put(argname, new TextWidget(arglabel, argdefault));
                                }
                                choicenames.clear();
                                choicevalues.clear();
                                argtype = null;
                                arglabel = null;
                                maximum = 0;
                                minimum = 500000;
                                argintvalue = 0;
                            }

                            ///// PERFORMANCE /////
                            // *** DO INSTEAD OF "arg".equals(fvalue) ***//
                            if (!saveMenuItem) {
                                // create a new argument parameter
                                argname = fvalue.toLowerCase();
                            } else {
                                argname = null;
                            }
                        }
                    }
                    //////////////////////////////
                    //**************************//
                    //* create a new menu item *//
                    //**************************//
                    //////////////////////////////
                    if (saveMenuItem) {
                        if (itemName != null) {
                            /* THIS BLOCK OF CODE
                             * Determines whether the program has widgets that require input.
                             * If there are no widgets the program will launch automatically when called.
                             */
                            if (argList.isEmpty() && helpCommand == null) {
                                if (runCommand != null) {
                                    itemAction = new CommandThread(GDE2BLArgs(argList, fileVariables, runCommand), widgets);
                                } else if (helpCommand != null) {
                                    itemAction = new CommandThread(GDE2BLArgs(argList, fileVariables, runCommand), widgets);
                                }
                            } else {
                                List<CommandButton> buttons = new LinkedList<CommandButton>();

                                if (runCommand != null) {
                                    buttons.add(new CommandButton(widgets, "Run", GDE2BLArgs(argList, fileVariables, runCommand), true));
                                }
                                if (helpCommand != null) {
                                    buttons.add(new CommandButton(widgets, "Help", GDE2BLArgs(argList, fileVariables, helpCommand), false));
                                }
                                itemAction = new RunWindow(itemName, buttons, widgets);
                            }
                            jmiresult = new JMenuItem(itemName);
                            jmiresult.addActionListener(itemAction);
                            BLMain.addMenuItem(menuName, jmiresult);
                            widgets = new LinkedHashMap<String, Widget>();
                        } else {
                            widgets.clear();
                        }

                        itemName = null;
                        fileVariables.clear();
                        argList.clear();
                        runCommand = null;
                        helpCommand = null;
                    } // only do processing if the line is not null
                    if (!"".equals(fname)) {
/////////////////////////////////
//*****************************//
//* create a new menu heading *//
//*****************************//
/////////////////////////////////
                        if ("menu".equals(fname)) {
                            menuName = fvalue;
                            BLMain.addMenuHeading(menuName);
                        } else if ("item".equals(fname)) {
                            itemName = fvalue;
                        } else if (itemName != null) {

////////////////////////////////////
//********************************//
//* do menu item data processing *//
//********************************//
////////////////////////////////////
// this is optional for improving the usability of the menu item
                            if (fname.startsWith("item")) {
                                if ("itemmeta".equals(fname) && fvalue.length() >= 1) {
                                    // generate the mnemonic for the menu item
                                    // this method allows only mnemonics of length 1
                                    // (menu mnemonics in java are all of length 1, except complex ones involving shift/ctrl/etc.)
                                    //if ((fvalue.charAt(0) >= 'a' && fvalue.charAt(0) <= 'z') || (fvalue.charAt(0) >= 'A' && fvalue.charAt(0) <= 'Z')) {
                                    //currentItem.putValue(javax.swing.Action.MNEMONIC_KEY, new Integer(javax.swing.KeyStroke.getKeyStroke(fieldValue.charAt(0)).getKeyCode()));
                                    //}
                                } else if ("itemlabel".equals(fname)) {
                                } else if ("itemopen".equals(fname)) {
                                } else if ("itemhelp".equals(fname)) {
                                    // obtain the help command for the menu item
                                    helpCommand = BLMain.getProperty("GDE.help.viewer") + " " + BLMain.envreplace(BLMain.getProperty("GDE.help.path")) + File.separator + fvalue;

                                } else if ("itemmethod".equals(fname)) {
                                    // get the command corresponding to the menu item
                                    runCommand = fvalue;
                                }

                            } else if (fname.startsWith("arg")) {
                                ////////////////////////////////////////
                                //************************************//
                                //* do argument parameter processing *//
                                //************************************//
                                ////////////////////////////////////////
                                // this is used to allow for parameter passing for running commands
                                if ("argtype".equals(fname)) {
                                    // changes the argument type
                                    argtype = WidgetType.toType(fvalue);
                                } else if ("arglabel".equals(fname)) {
                                    // changes the label of the command
                                    arglabel = fvalue;
                                } else if ("argmin".equals(fname)) {
                                    // changes the minimum value for the argument
                                    // this is applicable only to number choosers
                                    try {
                                        minimum = Integer.parseInt(fvalue);
                                    } catch (NumberFormatException nfe) {
                                        BLMain.error("Invalid GDE field value (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                                    }
                                } else if ("argmax".equals(fname)) {
                                    // changes the maximum value for the argument
                                    // this is applicable only to number choosers
                                    try {
                                        maximum = Integer.parseInt(fvalue);
                                    } catch (NumberFormatException nfe) {
                                        BLMain.error("Invalid GDE field value (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                                    }
                                } else if ("argvalue".equals(fname) || "argtext".equals(fname)) {
                                    // changes the default value for the argument
                                    argdefault = fvalue;
                                } else if ("argchoice".equals(fname)) {
                                    tmpindex = fvalue.indexOf(':');
                                    if (tmpindex >= 0) {
                                        // adds a new choice for the argument
                                        // this is applicable only to selection widgets
                                        choicenames.add(fvalue.substring(0, tmpindex));
                                        choicevalues.add(fvalue.substring(tmpindex + 1));
                                    } else {
                                        choicenames.add(fvalue);
                                        choicevalues.add("");
                                        BLMain.warning("Badly formed argchoice field: " + fvalue, "GDEMenu");
                                    }
                                }
                            } else if (fname.startsWith("in")) {
                                ///////////////////////////////////
                                //*******************************//
                                //* do I/O parameter processing *//
                                //*******************************//
                                ///////////////////////////////////
                                //***********************
                                //* Handle Input fields *
                                //***********************
                                if ("informat".equals(fname)) {
                                    informat = fvalue;
                                } else if ("inmask".equals(fname)) {
                                    BLMain.warning("Warning unsupported GDE menu field \"inmask\" -- ignoring", "GDEMenu");
                                } else if ("insave".equals(fname)) {
                                    insave = true;
                                }
                            } else if (fname.startsWith("out")) {
                                //************************
                                //* Handle Output fields *
                                //************************
                                if ("outformat".equals(fname)) {
                                    outformat = fvalue;
                                } else if ("outsave".equals(fname)) {
                                    outsave = true;
                                } else if ("outoverwrite".equals(fname)) {
                                    outoverwrite = true;
                                }
                            }
                        } else {
                            BLMain.error("Invalid GDE location or field value (item: " + itemName + "): " + fname + " = " + fvalue, "GDEMenu plugin");
                        }
                    }
                } while (line != null);

                // close the file
                reader.close();
            } catch (Throwable e) {
                e.printStackTrace(System.err);
            }
        }
    }

    private static String GDE2BLArgs(Set<String> argList, List<String> fileVariables, String runCommand) {
        int start = -1;
        int end = -1;
        String name = "";
        String value = "";
        String smallrun = runCommand.toLowerCase();
        String test = null;

        // create the command string (NOTE: variable.toString().replaceAll(Pattern.quote("$"), "\\\\\\$") is used to prevent regex grouping (e.g. $0, etc.))
        while ((start = runCommand.indexOf('$', start)) >= 0) {
            end = start + 1;
            while (end < runCommand.length() && (BLMain.isDigit(runCommand.charAt(end)) || runCommand.charAt(end) == '_')) {
                end++;
            }
            test = runCommand.substring(start + 1, end).toLowerCase();
            if (argList.contains(test)) {
                runCommand = (start == 0 ? "" : runCommand.substring(0, start)) + '%' + test.toLowerCase() + '%' + (end > runCommand.length() ? "" : runCommand.substring(end));
            }
            start++;
        }

        smallrun = runCommand.toLowerCase();
        for (String file : fileVariables) {
            int searchCount = 0;
            // reset the start and positions
            start = -1;
            end = -1;

            // make sure name is valid
            if (file != null && !"".equals(file)) {

                // itterate through and find each instance of the name to replace
                while ((start = smallrun.indexOf(file, end + 1)) >= 0) {
                    end = start + file.length();

                    // make sure that the name is not part of a bigger name
                    if ((start == 0 || !BLMain.isDigit(runCommand.charAt(start - 1))) && (end >= runCommand.length() || !BLMain.isDigit(runCommand.charAt(end)))) {
                        runCommand = (start == 0 ? "" : runCommand.substring(0, start)) + '%' + file.toLowerCase() + '%' + (end > runCommand.length() ? "" : runCommand.substring(end));
                        smallrun = runCommand.toLowerCase();
                    }
                }
            }
        }
        return runCommand;
    }
}
