package org.biolegato.gdesupport.canvas;

/*
 * BLGDECanvas.java
 *
 * Created on December 3, 2007, 1:56 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.io.File;
import java.io.FileWriter;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import org.biolegato.gdesupport.data.Seq;
import org.biolegato.gdesupport.data.Dataset;
import org.biolegato.main.BLMain;
import org.biolegato.main.DataCanvas;
import org.biolegato.gdesupport.files.DataFormat;

/**
 * The GDE-style sequence-based canvas class
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public final class GDECanvas implements DataCanvas, ListSelectionListener {

    /**
     * The action "Open..."
     */
    public final AbstractAction openAction = new AbstractAction("Open...") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_O));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {           /* Event handler - open the file */
            JFileChooser openDialog = new JFileChooser();
            File[] openFiles;

            openDialog.setCurrentDirectory(BLMain.getCurrentPWD());
            openDialog.setAcceptAllFileFilterUsed(true);
            openDialog.setMultiSelectionEnabled(true);

            // itterate through all file format subclasses
            for (DataFormat currentFormat : DataFormat.FORMAT_LIST) {
                // add the file filter to the JFileChooser
                openDialog.addChoosableFileFilter(currentFormat);
            }

            // if a file is selected, open it
            if (openDialog.showOpenDialog(BLMain.getJFrame())
                    == JFileChooser.APPROVE_OPTION) {
                if (openDialog.getSelectedFiles() != null) {
                    openFiles = openDialog.getSelectedFiles();
                    for (File ofile : openFiles) {
                        if (ofile.exists() && ofile.isFile()) {
                            try {
                                if (openDialog.getFileFilter() != null && !openDialog.getFileFilter().equals(
                                        openDialog.getAcceptAllFileFilter())) {
                                    DataFormat.readFile(datamodel, (DataFormat) openDialog.getFileFilter(), ofile);
                                } else {
                                    DataFormat.readFile(datamodel, ofile);
                                }
                            } catch (Throwable e) {
                                e.printStackTrace(System.err);
                            }
                        }
                    }
                }
                if (openDialog.getCurrentDirectory() != null) {
                    BLMain.setCurrentPWD(openDialog.getCurrentDirectory());
                }
            }
        }
    };
    /**
     * The action "Save As..."
     */
    public final AbstractAction saveAsAction = new AbstractAction("Save As...") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_S));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - save the file */
            JFileChooser saveDialog = new JFileChooser();

            saveDialog.setCurrentDirectory(BLMain.getCurrentPWD());
            saveDialog.setAcceptAllFileFilterUsed(false);

            // itterate through all file format subclasses
            for (DataFormat currentFormat : DataFormat.FORMAT_LIST) {
                // add the file filter to the JFileChooser
                saveDialog.addChoosableFileFilter(currentFormat);
            }

            // if a file is selected, save to it
            if (saveDialog.showSaveDialog(BLMain.getJFrame())
                    == JFileChooser.APPROVE_OPTION
                    && saveDialog.getSelectedFile() != null
                    && (!saveDialog.getSelectedFile().exists()
                    || javax.swing.JOptionPane.showConfirmDialog(null,
                    "Overwrite existing file?", "Overwrite", javax.swing.JOptionPane.OK_CANCEL_OPTION,
                    javax.swing.JOptionPane.QUESTION_MESSAGE) != javax.swing.JOptionPane.CANCEL_OPTION)) {
                // write the file
                try {
                    DataFormat.writeAll(datamodel, (DataFormat) saveDialog.getFileFilter(), saveDialog.getSelectedFile());
                } catch (Throwable e) {
                    e.printStackTrace(System.err);
                }
                if (saveDialog.getCurrentDirectory() != null) {
                    BLMain.setCurrentPWD(saveDialog.getCurrentDirectory());
                }
            }
        }
    };
    /**
     * The action "Properties..."
     */
    public final AbstractAction propertiesAction = new AbstractAction("Properties...") {

        private static final long serialVersionUID = 7526472295622776157L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - open canvas properties window */
            new GDECanvasProperties(BLMain.getJFrame(), canvasSelf);
        }
    };
    /**
     * Action for splitting the canvas
     */
    private final AbstractAction splitAction = new AbstractAction("Split") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed(ActionEvent e) {

            altDataCollector = new GDETextArea(canvasSelf, datamodel);
            canvasPane.remove(dataPane);

            altDataPane = new JScrollPane(altDataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
            splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, dataPane, altDataPane);
            splitPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
            splitPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);
            canvasPane.add(splitPane);

            dataCollector.popup.remove(splitMenuItem);
            dataCollector.popup.add(joinMenuItem);

            altDataCollector.popup.add(new JMenuItem(joinAction));
        }
    };
    /**
     * Action for joining split canvases
     */
    public final AbstractAction joinAction = new AbstractAction("Join") {

        private static final long serialVersionUID = 7526472295622777032L;

        public void actionPerformed(ActionEvent e) {
            if (currentPane == altDataCollector) {
                currentPane = dataCollector;
            }
            if (splitPane != null) {
                canvasPane.remove(splitPane);
                splitPane = null;
                altDataPane = null;
                canvasPane.add(dataPane);
            }

            dataCollector.popup.remove(joinMenuItem);
            dataCollector.popup.add(splitMenuItem);
        }
    };
    /**
     * The action "Cut"
     */
    public final AbstractAction cutAction = new AbstractAction("Cut") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_C));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            currentPane.copyClipboard();
            currentPane.deleteSelection();
        }
    };
    /**
     * The action "Copy"
     */
    public final AbstractAction copyAction = new AbstractAction("Copy") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_Y));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            currentPane.copyClipboard();
        }
    };
    /**
     * The action "Paste"
     */
    public final AbstractAction pasteAction = new AbstractAction("Paste") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_P));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            currentPane.paste();
        }
    };
    /**
     * The action "Change Case"
     */
    public final AbstractAction changeCaseAction = new AbstractAction("Change case") {

        private static final long serialVersionUID = 7526472295622777033L;	    /* Serialization number - required for no warnings*/


        {
            putValue(MNEMONIC_KEY, new Integer(java.awt.event.KeyEvent.VK_E));
        }  /* Sets the mnemonic for the event */


        public void actionPerformed(java.awt.event.ActionEvent evt) {	    /* Event handler - exit the program */
            currentPane.changeCase();
        }
    };
    /**
     * The current font for the GDE canvas
     */
    Font currentFont = DEFAULT_FONT;
    /**
     * The main canvas display
     */
    public Box mainDisplay = new Box(BoxLayout.PAGE_AXIS);
    /**
     * The current dataset for the GDE canvas
     */
    private final Dataset datamodel = new Dataset();
    /**
     * Used for selecting genes
     */
    private GDEList nameList = new GDEList(this, datamodel);
    /**
     * The text area for data manipulation
     */
    private GDETextArea dataCollector = null;
    /**
     * The alternate text area for data manipulation (for splits)
     */
    private GDETextArea altDataCollector = null;
    /**
     * Stores which canvas contains the cursor and any current data selections.
     */
    private GDECanvasObject currentPane = nameList;
    /**
     * The scroll pane containing the main canvas
     */
    private JScrollPane dataPane = null;
    /**
     * The scroll pane containing the split canvas
     */
    private JScrollPane altDataPane = null;
    /**
     * The box to store all split canvases
     */
    private Box canvasPane = new Box(BoxLayout.LINE_AXIS);
    /**
     * The split pane for split canvases
     */
    private JSplitPane splitPane = null;
    /**
     * Reference pointer to self
     */
    private final GDECanvas canvasSelf = this;
    /**
     * Stores the row and column position status
     */
    private final JLabel status = new JLabel("Row: 1 Col: 1");
    /**
     * Stores the insertion mode status
     */
    private final JLabel insertStatus = new JLabel("     ");
    /**
     * The menu item "Select sequence by name"
     */
    private final JMenuItem selectByNameMenuItem = new JMenuItem(nameList.selectByNameAction);
    /**
     * Action for splitting the canvas
     */
    private final JMenuItem splitMenuItem = new JMenuItem(splitAction);
    /**
     * The menu item for joining split canvases
     */
    private final JMenuItem joinMenuItem = new JMenuItem(joinAction);
    /**
     * The menu item for grouping sequences in the canvas
     */
    private final JMenuItem groupMenuItem = new JMenuItem(nameList.groupAction);
    /**
     * The menu item for ungrouping sequences in the canvas
     */
    private final JMenuItem ungroupMenuItem = new JMenuItem(nameList.ungroupAction);
    /**
     * The menu item for "Get Info..."
     */
    private final JMenuItem getInfoMenuItem = new JMenuItem(nameList.getInfoAction);
    /**
     * The default font for the canvas
     */
    public static final Font DEFAULT_FONT = new Font("Monospaced", Font.PLAIN, 12);
    /**
     * Used for serialization purposes
     */
    private static final long serialVersionUID = 7526472295622776154L;

    /**
     * Creates a new instance of BLGDECanvas
     */
    public GDECanvas() {
        int menuInsertCount = 0;

        //////////////////////////////////////
        //**********************************//
        //* ADD THE DEFAULT TOP MENU ITEMS *//
        //**********************************//
        //////////////////////////////////////

        /*************************
         * Add the "Open" button *
         *************************/
        BLMain.addMenuItem(0, "File", new JMenuItem(openAction));

        /*******************************
         * Add the "Save As..." button *
         *******************************/
        BLMain.addMenuItem(1, "File", new JMenuItem(saveAsAction));

        /*******************************
         * Add the "Properties" button *
         *******************************/
        BLMain.addMenuItem(2, "File", new JMenuItem(propertiesAction));

        // NOTE: count++ increments count after retrieving its value for use in the below functions
        // this choice of count++ allows variable length menus or simple re-ordering of choices
        BLMain.addMenuHeading(1, "Edit").insert(new JMenuItem(cutAction), menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(new JMenuItem(copyAction), menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(new JMenuItem(pasteAction), menuInsertCount++);
        // menu items for name list
        BLMain.addMenuHeading(1, "Edit").insert(groupMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(ungroupMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(getInfoMenuItem, menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(new JMenuItem(nameList.selectGroupAction), menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(new JMenuItem(nameList.selectAllAction), menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(new JMenuItem(changeCaseAction), menuInsertCount++);
        BLMain.addMenuHeading(1, "Edit").insert(selectByNameMenuItem, menuInsertCount++);

        // create new objects
        dataCollector = new GDETextArea(canvasSelf, datamodel);

        // setup listeners
        nameList.addListSelectionListener(canvasSelf);

        // add items to the textarea's popup menu
        dataCollector.popup.add(splitMenuItem);

        // create the scroll panes
        dataPane = new JScrollPane(dataCollector, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        JScrollPane listPane = new JScrollPane(nameList, JScrollPane.VERTICAL_SCROLLBAR_NEVER, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        dataPane.setPreferredSize(new Dimension(200, 150));
        listPane.setPreferredSize(new Dimension(100, 150));

        // make the vertical scroll bars the same for both scroll panes
        dataPane.setVerticalScrollBar(listPane.getVerticalScrollBar());

        // add the dataPane to the canvasPane
        canvasPane.add(dataPane);

        // create the main split pane
        JSplitPane mainPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, listPane, canvasPane);
        mainPane.setAlignmentX(JSplitPane.CENTER_ALIGNMENT);
        mainPane.setAlignmentY(JSplitPane.CENTER_ALIGNMENT);

        // create the status bar
        Box statusBar = new Box(BoxLayout.LINE_AXIS);
        statusBar.setAlignmentX(Box.LEFT_ALIGNMENT);
        statusBar.add(status);
        statusBar.add(insertStatus);

        // add the panes to the canvas
        mainDisplay.add(mainPane, BorderLayout.PAGE_START);
        mainDisplay.add(new JSeparator(SwingConstants.HORIZONTAL));
        mainDisplay.add(statusBar, BorderLayout.PAGE_END);
        
        updateFont(12, false);
    }

    /**
     * Returns the name to display in the canvas tab for
     **
     * @return  "GDE"
     */
    public String getName() {
        return "GDE";
    }

    /**
     * Receives cursor updates for use in the status bar.
     **
     * @param column the new column of the cursor.
     * @param row the new row of the cursor.
     */
    public void cursorChange(int column, int row) {
        status.setText("Row: " + (row + 1) + " Col: " + (column + 1));
    }

    /**
     * Receives insertion mode change updates for use in the status bar.
     *
     * @param mode the new insertion mode status.
     */
    public void insertionMode(boolean mode) {
        if (mode) {
            insertStatus.setText("[INS]");
        } else {
            insertStatus.setText("     ");
        }
    }

    /**
     * Clears the data collector when an item is selected from the sequence list
     * @param e currently ignored
     */
    public void valueChanged(ListSelectionEvent e) {
        if (currentPane != nameList) {
            currentPane = nameList;
            dataCollector.clearSelection();
            if (altDataCollector != null) {
                altDataCollector.clearSelection();
            }
            BLMain.addMenuHeading(1, "Edit").insert(groupMenuItem, 3);
            BLMain.addMenuHeading(1, "Edit").insert(ungroupMenuItem, 4);
            BLMain.addMenuHeading(1, "Edit").insert(getInfoMenuItem, 5);
        }
    }

    /**
     * Manages mutual exclusion between the textarea and the list
     *
     * @param source the source of the event
     */
    public void selectionMade(GDETextArea source) {
        if (source != null && currentPane != source) {
            currentPane = source;
            nameList.clearSelection();
            if (altDataCollector != null && source != altDataCollector) {
                altDataCollector.clearSelection();
            }
            if (source != dataCollector) {
                dataCollector.clearSelection();
            }
            BLMain.addMenuHeading(1, "Edit").remove(groupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(ungroupMenuItem);
            BLMain.addMenuHeading(1, "Edit").remove(getInfoMenuItem);
        }
    }

    /**
     * Updates the font for the GDE canvas and its sub-objects
     *
     * @param size the new font size
     * @param bold whether to bold the font
     */
    public final void updateFont(int size, boolean bold) {
        currentFont = currentFont.deriveFont((bold ? Font.BOLD : Font.PLAIN), size);
        dataCollector.setFont(currentFont);
        nameList.setFont(currentFont);
        if (altDataCollector != null) {
            altDataCollector.setFont(currentFont);
        }
    }

    /**
     * Writes a file out from the canvas
     **
     * @param formatString the file format to use for writing the file.
     * @param file the file to write out.
     */
    public void writeFile(String formatString, File file) {
        Seq line;
        DataFormat format = DataFormat.getFormat(formatString);
        FileWriter writer = null;

        // write the file
        try {
            if (format != null && currentPane != null) {
                writer = new FileWriter(file);   // the file writer object to write the file with.

                currentPane.writeOut(format, writer);

                // flush and close the file writer buffer.
                writer.flush();
                writer.close();
            } else {
                BLMain.error("Unsupported file format", "GDECanvas");
            }
        } catch (Throwable th) {
            th.printStackTrace(System.err);
        }
    }
    
    /**
     * Reads a file into the canvas
     **
     * @param formatString the file format to use for parsing the file.
     * @param file the file to read in.
     */
    public void readFile(String formatString, File file) {
        DataFormat.readFile(datamodel, DataFormat.getFormat(formatString), file);
    }

    /**
     * Displays the canvas
     **
     * @return a component representing the canvas (visually)
     */
    public Component display () {
        return mainDisplay;
    }

}
