/*
 * DataFormat.java
 *
 * Created on January 30, 2008, 11:54 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.biolegato.gdesupport.files;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.MappedByteBuffer;
import org.biolegato.gdesupport.data.Seq;
import javax.swing.filechooser.FileFilter;
import org.biolegato.gdesupport.data.Dataset;

/**
 * Class to represent reading of data formats.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public abstract class DataFormat extends FileFilter {

    public static final DataFormat GENBANK = new GenBankFile2008();
    public static final DataFormat GDEFILE = new GDEFile();
    public static final DataFormat GDEFLAT = new GDEFlatfile();
    public static final DataFormat FASTA = new FastAFile();
    /**
     * The list of available file formats
     */
    public static final DataFormat[] FORMAT_LIST = new DataFormat[]{
        GENBANK, GDEFILE, GDEFLAT, FASTA
    };

    /**
     * Creates a new instance of DataFormat
     */
    public DataFormat() {
    }

    /**
     * Obtains the name of the file format for File Chooser purposes
     *
     * @return the name of the file format
     */
    public String getName() {
        return "";
    }

    /**
     * Translates a string from the BioLegato internal format to the given file format.
     *
     * @param data the string to convert
     * @return the resulting string
     */
    public abstract void translateTo(Appendable destination, Seq data, int offset, int length) throws IOException;

    /**
     * Translates a string from the given file format into the BioLegato internal format.
     *
     * @param data the string to convert.
     * @return the resulting string.
     * @throws IOException any exeptions that occur while reading the stream are passed.
     */
    public abstract void translateFrom(Dataset datamodel, java.io.BufferedReader data) throws IOException;

    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the string to test
     * @return whether the format is correct
     */
    public abstract boolean isFormat(java.io.Reader test);

    /**
     * Used to auto-detect file formats.
     *
     * @param data the string to convert
     * @return The resulting sequences.
     */
    public static DataFormat autodetect(java.io.BufferedReader data) {
        int count = 0;
        DataFormat result = null;

        // ensure the data we are using is not null
        if (data != null) {
            // itterate through all file formats
            while (count < FORMAT_LIST.length && !FORMAT_LIST[count].isFormat(data)) {
                count++;
            }
            if (count < FORMAT_LIST.length) {
                result = FORMAT_LIST[count];
            }
        }
        return result;
    }

////////////////////////
//********************//
//* STATIC FUNCTIONS *//
//********************//
////////////////////////
    /**
    /**
     * Finds the filetype that corresponds to the given hashname (null if not successful).
     *
     * @param hashname the hashname to search for.
     * @return The result of the search (returns null on failure).
     */
    public static DataFormat getFormat(String hashname) {
        DataFormat found = DataFormat.GDEFLAT;  // the data format found (null if not found).

        // ensure the name we are searching for is not null
        if ("genbank".equals(hashname)) {
            found = DataFormat.GENBANK;
        } else if ("gde".equals(hashname)) {
            found = DataFormat.GDEFILE;
        } else if ("fasta".equals(hashname)) {
            found = DataFormat.FASTA;
        }
        return found;
    }

    /**
     * Reads a file into the canvas
     **
     * @param format the file format to use for parsing the file.
     * @param currentFile the file to read in.
     */
    /**
     * Reads a file into the canvas
     **
     * @param format the file format to use for parsing the file.
     * @param currentFile the file to read in.
     */
    public static void readFile(Dataset datamodel, File file) throws IOException {
        /*
        TODO: possible faster method to read big files.
        -----------------------------------------------
        stream = new FileInputStream(file.getPath()).getChannel().map(FileChannel.MapMode.READ_ONLY, 0, file.length());
        result = translateFrom(new LineNumberReader(stream));
         */
        MappedByteBuffer stream = null;
        BufferedReader reader = new BufferedReader(new InputStreamReader(new BufferedInputStream(new FileInputStream(file))));
        DataFormat format = DataFormat.autodetect(reader);

        if (format != null && file != null && file.exists() && file.isFile() && file.length() > 0) {
            readFile(datamodel, format, reader);
        }
    }

    /**
     * Reads a file into the canvas
     **
     * @param format the file format to use for parsing the file.
     * @param currentFile the file to read in.
     */
    public static void readFile(Dataset datamodel, DataFormat format, File file) {
        MappedByteBuffer stream = null;
        BufferedReader reader = null;

        if (format != null && file != null && file.exists() && file.isFile() && file.length() > 0) {
            try {
                /*
                TODO: possible faster method to read big files.
                -----------------------------------------------
                stream = new FileInputStream(file.getPath()).getChannel().map(FileChannel.MapMode.READ_ONLY, 0, file.length());
                result = translateFrom(new LineNumberReader(stream));
                 */
                reader = new BufferedReader(new InputStreamReader(new BufferedInputStream(new FileInputStream(file))));
                readFile(datamodel, format, reader);
            } catch (IOException ioe) {
                ioe.printStackTrace(System.err);
            }
        }
    }

    /**
     * Reads a file into the canvas
     **
     * @param format the file format to use for parsing the file.
     * @param currentFile the file to read in.
     */
    private static void readFile(Dataset datamodel, DataFormat format, BufferedReader buffer) throws IOException {
        format.translateFrom(datamodel, buffer);
    }

    /**
     * Writes a list of sequences to a file.
     *
     * @param file the file to writer to.
     * @param data the contents to writer.
     * @throws IOException throws any exceptions that occur while writing to the file
     */
    public static void writeAll(Dataset datamodel, DataFormat format, File file) throws IOException {
        final int maxlines = datamodel.getSize();
        Seq current;
        FileWriter writer = new FileWriter(file);   // the file writer object to write the file with.

        // translate and write the file.
        for (int count = 0; count < maxlines; count++) {
            current = datamodel.getLine(count);
            format.translateTo(writer, current, 0, current.getSequence().length());
        }

        // flush and close the file writer buffer.
        writer.flush();
        writer.close();
    }
}
