package org.biolegato.gdesupport.files;
/*
 * GDEFlatfile.java
 *
 * Created on January 30, 2008, 11:58 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

import org.biolegato.gdesupport.data.Seq;
import org.biolegato.gdesupport.data.Seq.Type;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import org.biolegato.gdesupport.data.Dataset;

/**
 * This class acts as a parser/translator for GDE flat files.
 *
 * @author Graham Alvare
 * @author Brian Fristensky
 */
public class GDEFlatfile extends DataFormat {

    /**
     * Creates a new instance of GDEFlatfile
     */
    public GDEFlatfile() {
    }

    /**
     * Translates a sequence into the GDE flat file format.
     *
     * @param seq the sequence to translate
     * @return the resulting string
     */
    public void translateTo(Appendable result, Seq seq, int offset, int length) throws IOException {
        char type = '#';

        // translate the data
        if (seq != null) {
            type = typeToFlatFile((Type) seq.getType());
            if (type != '%' && type != '@' && type != '\"') {
                type = '#';
            }
            // NOTE: append is faster than + or concat operators
            result.append(type).append(seq.getName()).append("\n").append(seq.getSequence(), offset, length).append("\n");
        }
    }

    /**
     * Translates data in the GDE flat file format to sequence objecte
     *
     * @param data the buffered reader to parse
     * @return the translated sequences
     * @throws IOException any exeptions that occur while reading the stream are passed
     */
    public void translateFrom(Dataset datamodel, java.io.BufferedReader data) throws
            IOException {
        int y = datamodel.getSize();
        String line = "";
        Seq.Type type = Type.DNA;
        String name = "";
        StringBuffer sequence = null;

        while ((line = data.readLine()) != null) {
            if (line.startsWith("#") || line.startsWith("%") || line.startsWith(
                    "@") || line.startsWith("\"")) {
                if (line.indexOf(' ') > 0) {
                    line = line.substring(0, line.indexOf(' '));
                }
                if (line.indexOf('\t') > 0) {
                    line = line.substring(0, line.indexOf('\t'));
                }
                if (line.indexOf('\n') > 0) {
                    line = line.substring(0, line.indexOf('\n'));
                }
                if (line.indexOf('\r') > 0) {
                    line = line.substring(0, line.indexOf('\r'));
                }
                if (sequence != null) {
                    datamodel.addSequence(y, new Seq(type, name, sequence));
                    y++;
                }
                type = flatFileToType(line.trim().charAt(0));
                name = line.trim().substring(1);
                sequence = new StringBuffer();
            } else if (sequence != null) {
                sequence.append(line.trim());
            }
        }
        if (sequence != null) {
            datamodel.addSequence(y, new Seq(type, name, sequence));
            y++;
        }
    }

    /**
     * Determines whether a specified file is of type GDE flat file (based on extension).
     * Currently the only extension supported is ".flat".
     *
     * @param file the file to test
     * @return true if the file is of type GDE flat file (otherwise false)
     * @see javax.swing.filechooser.FileFilter#accept
     */
    public boolean accept(File file) {
        return (file.isDirectory() || file.getAbsolutePath().toLowerCase().
                endsWith(".flat"));
    }

    /**
     * Returns a description of the file format that can be displayed to the user.
     *
     * @return the string description of the file format
     * @see javax.swing.filechooser.FileFilter#getDescription
     */
    public String getDescription() {
        return "GDE Flatfile (*.flat)";
    }

    /**
     * Converts a GDE flat file type character into a Cell.Type structure.
     * 
     * 
     * 
     * 
     * @param test the character to test for type
     * @return the resulting type
     */
    private static Type flatFileToType(char test) {
        Type result = Type.DNA;
        switch (test) {
            case '%':
                result = Type.PROTEIN;
                break;
            case '@':
                result = Type.MASK;
                break;
            case '\"':
                result = Type.TEXT;
                break;
        }
        return result;
    }

    /**
     * Converts a sequence's type enum to the GDE flat file format.
     *
     * @param type the type enum to convert
     * @return the GDE flat file equivilent
     */
    private static char typeToFlatFile(Type t) {
        char result = '#';
        switch (t) {
            case PROTEIN:
                result = '%';
                break;
            case MASK:
                result = '@';
                break;
            case TEXT:
                result = '\"';
                break;
        }
        return result;
    }

    /**
     * Used to auto-detect Bio Legato formats
     *
     * @param test the reader to parse data from
     * @return whether the format is correct
     */
    @Override
    public boolean isFormat(Reader test) {
        int check = ' ';

        try {
            while (check != -1 && test.ready() && (check == ' ' || check == '\t' || check == '\n' || check
                    == '\r')) {
                test.mark(2);
                check = test.read();
            }
            test.reset();
        } catch (Throwable e) {
            e.printStackTrace(System.err);
        }
        return (check == '#' || check == '%' || check == '@' || check == '\"');
    }
}
