//=====================================================================
// File:    BinAnalysis.java
// Class:   BinAnalysis
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

/**
 * This class is an <code>AnalysisOp</code> which outputs information 
 * about all of the bins in the gel. For each bin in the gel, the
 * name, if defined is printed out, as well as the location and range.
 * Then all of the overall scoring information is displayed for the bin.
 * Finally, the label assigned to each lane in the gel is printed out.
 *
 * <p> Sample output for a single bin (text enclosed in [] may not always
 * be printed):
 * <pre> [BinName at] 102.4 +/- 2.3 bp
 * Mean: 101.9 StdDev: 0.02 #A: 10 #B 12 from 10/10 peaks.
 * ABABBAABBABAABABBABBBA
 * </pre>
 * or in general:
 * <pre>
 * [<i>name</i> at] <i>location</i>  +/- <i>range</i>
 * <i>overall score info line 1</i>
 * <i>overall score info line 2</i>
 * .
 * <i>overall score info line n</i>
 * <i>lane1Label|lane2Label|....|laneNLabel</i>
 * </pre>
 * <p> Lines in the output string are seperated with '\n'
 *
 * @see Bin
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public class BinAnalysis extends AnalysisOp
{
  /**
   * Creates a new BinAnalysis object.
   */
  public BinAnalysis()
  {
    name = "Bin Display";
    descript = "Display the scoring of all of the bins.";
    helpFile = "bindisplay.html"; 
  }

  /**
   * Provides an anlysis for the the specified gel. See the class
   * description for the details of the analysis.
   *
   * @param gel   the gel to analyze
   */
  public String analyze(Gel gel)
  {
    StringBuffer output = new StringBuffer();
    DataList lanes = gel.getLanes();
    DataList bins = gel.getBins();

    Bin bin;
    Lane ln;
    String scoreInfo[];
    for(int i=0; i < bins.size(); i++)
      {
     bin = (Bin) bins.dataAt(i);
     
     // score the bin if neccessary
     if(!bin.isScored())
       bin.score(lanes);

     if(!bin.getName().equals(""))
       {
         output.append(bin.getName() + " at ");
       }
     output.append(bin.getLocation() + " +/- " +
                bin.getRange() + "\n");
     
     // add the score info.
     scoreInfo = bin.getScoreInfo();
     for(int j=0; j < scoreInfo.length; j++)
       {
         output.append(scoreInfo[j] + "\n");
       }

     // print the scoring for each lane
     for(int j=0; j < lanes.size(); j++)
       {
         output.append(bin.getScore( (Lane) lanes.dataAt(j)));
       }
     
     // go to the next one
     output.append("\n\n");
     }

    return new String(output);
  }

  /**
   * No options for this operation, so <code>null</code> is returned.
   *
   * @return <code>null</code>
   */
  public Option[] getOptions()
  {
    return null;
  }

  /**
   * Since there are no options for this operation, this method will
   * do absolutely nothing.
   */
  public void setOptions(Option[] opts)
  {
    // nothing to set.
  }

  /**
   * Gives the name of this bin analysis operation
   *
   * @return the name, which is "Bin Display".
   */
  public String getName()
  {
    return name;
  }

  /**
   * Gives a one sentence description of this analysis operation.
   *
   * @return the description
   */
  public String getDescription()
  {
    return descript;
  }

  /**
   * Gives a file that is the help file for this analysis operation
   *
   * @return a plaintext of html file containing help information
   */
  public String getHelpFile()
  {
    return helpFile;
  }
}
