//=====================================================================
// File:    MapMakerAnalysis.java
// Class:   MapMakerAnalysis
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

/**
 * This class is an <code>AnalysisOp</code> which outputs information 
 * about all of the bins in the gel. For each bin in the gel, the
 * name, if defined is printed out. If the name is not printed out, then
 * a name is produced by adding 'n' to the front of the location, which
 * is rounded to a whole number. Each name is prefaced with a '*'. 
 * Next, the label assigned to each lane in the gel is printed out.
 *
 * <p> Sample output for two bin, one without a defined name and one with:
 * <pre> *n102
 * ABABBAABBABAABABBABBBA
 * 
 * *binName
 * ABBABABAAABBBBAABABABB
 * </pre>
 * or in general:
 * <pre>
 * *<i>name</i> or *n<i>location</i>
 * <i>lane1Label|lane2Label|....|laneNLabel</i>
 * </pre>
 *
 * <p> Lines in the output string are seperated with '\n'
 *
 * @see Bin
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public class MapMakerAnalysis extends AnalysisOp
{
  /**
   * Creates a new MapMakerAnalysis object.
   */
  public MapMakerAnalysis()
  {
    name = "MapMaker Compatible";
    descript = "Display the scoring of all of the bins in a format " + 
      "compatible with MapMaker.";
    helpFile = "mapanalysis.html";
  }

  /**
   * Provides an anlysis for the the specified gel. See the class
   * description for the details of the analysis.
   *
   * @param gel   the gel to analyze
   */
  public String analyze(Gel gel)
  {
    StringBuffer output = new StringBuffer();
    DataList lanes = gel.getLanes();
    DataList bins = gel.getBins();

    Bin bin;
    Lane ln;
    String scoreInfo[];
    for(int i=0; i < bins.size(); i++)
      {
	bin = (Bin) bins.dataAt(i);
	
	// score the bin if neccessary
	if(!bin.isScored())
	  bin.score(lanes);

	String name;

	// give the bin a name
	if(bin.getName().equals(""))
	  name = "*n" + Math.round(bin.getLocation());
	else
	  name = "*" + bin.getName();

	// check the length, it must be 8 charachters or less. This
	// does not include the '*' starting the name.
	if(name.length() > 9)
	  name = name.substring(0, 9);

	output.append(name + "\n");

	// print the scoring for each lane
	for(int j=0; j < lanes.size(); j++)
	  {
	    output.append(bin.getScore( (Lane) lanes.dataAt(j)));
	  }
	
	// go to the next one
	output.append("\n\n");
     }

    return new String(output);
  }

  /**
   * No options for this operation, so <code>null</code> is returned.
   *
   * @return <code>null</code>
   */
  public Option[] getOptions()
  {
    return null;
  }

 /**
   * Since there are no options for this operation, this method will
   * do absolutely nothing.
   */
  public void setOptions(Option[] opts)
  {
    // nothing to set.
  }

  /**
   * Gives the name of this bin analysis operation
   *
   * @return the name, which is "MapMaker Compatible".
   */
  public String getName()
  {
    return name;
  }

  /**
   * Gives a one sentence description of this analysis operation.
   *
   * @return the description
   */
  public String getDescription()
  {
    return descript;
  }

  /**
   * Gives a file that is the help file for this analysis operation
   *
   * @return a plaintext of html file containing help information
   */
  public String getHelpFile()
  {
    return helpFile;
  }
}
