//=====================================================================
// File:    Operation.java
// Class:   Operation
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

/**
 * This is an abstract class that should be extended to add functionality
 * to the program. The <code>Manager</code> subclasses expect objects of this
 * type. The <code>Manager</code> provides an easy way to access the different
 * operations, so most of the program does not have to know what operations
 * are available, it can find this out by quering a manager. The operation
 * itself contains information like a name and description. It also
 * contains a list of options that the operation can accept. However,
 * this class should be extended so the program can know more about the
 * operation. For example, <code>ImportFilter</code> extends this to 
 * read in different files.
 *
 * @see Manager
 * @see Option
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public abstract class Operation
{
  protected String name;         // the name of this operation
  protected String descript;     // a brief description
  protected String helpFile;     // represents the file that contains
                                 // the help info for this operation.
  protected Option[] options;

  /**
   * Access the name of the operation. It should be somewhat unique, short,
   * and descriptive. It's recommened to store the name internally in the
   * <code>name</code> variable.
   *
   * @return name of the operation
   */
  public abstract String getName();

  /**
   * Retrieves a short, approximately one sentence, description of the 
   * operation.
   *
   * @return the description
   */
  public abstract String getDescription();

  /**
   * Each operation should have a help file that describes its actions, types 
   * of files it works on, and an explination of any options that the filter
   * supports.
   *
   * @return Filename that contains the help information, either html or
   *   plaintext.
   */
  public abstract String getHelpFile();

  /**
   * Retrieves the options associated with this operation. In some cases,
   * these may be required parameters, but this method provides a way for
   * other classes to query this object and determine it's requirements.
   *
   * @return the options, <code>null</code> if there are none.
   */
  public abstract Option[] getOptions();

  /**
   * This is used to set the options to the specified values. These may
   * be required before a operation can be performed.
   *
   * @param opts  the values for the options that this operation
   *      understands.
   */
  public abstract void setOptions(Option[] opts);

}
