//=====================================================================
// File:    SizeFunction.java
// Class:   SizeFunction
// Package: AFLPcore
//
// Author:  James J. Benham
// Date:    August 10, 1998
// Contact: james_benham@hmc.edu
//
// Genographer v1.0 - Computer assisted scoring of gels.
// Copyright (C) 1998  Montana State University
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; version 2
// of the License.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// The GNU General Public License is distributed in the file GPL
//=====================================================================

package AFLPcore;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/** 
 * This is an abstract class that is used by <code>Lane</code> and
 * anything else that needs to convert back and forth between scan
 * numbers and sizes, in bp. Classes should extend this class to
 * provide a sizing function for the Lane and other classes. The first
 * entry in a Lane should be scan number 0.
 *
 * @see Lane
 * @see Operation
 *
 * @author James J. Benham
 * @version 1.0.0
 * @date August 10, 1998
 */

public abstract class SizeFunction extends Operation implements Cloneable
{
  /**
   * This should be used to initialize the sizing function with a set
   * of known peaks. These known points can then be used by the sizing 
   * function to compute whatever formulas it needs.
   *
   * @param standardPoints  A list of peaks that consist of the known
   *        points. <b>Note:</b> the area of the peak should contain
   *        the scan number of the peak.
   */
  public abstract void init(DataList standardPoints);

  /**
   * Gives the size that corresponds to the scan number using some
   * sort of size curve.
   *
   * @param scan  this should be the same as the index into the trace array
   *
   * @return the size in bp
   */
  public abstract double getSize(int scan);

  /**
   * Gives the index into the trace array, which should correspond to the
   * scan number for any given size.
   *
   * @param size the size in bp
   *
   * @return the scan number
   */
  public abstract int getScan(double size);

  /**
   * Sets the maximum scan number to be returned to the specified value.
   * this is useful because some methods may to be able to uniquely 
   * determine the scan number, but if one of the possibilities is beyond
   * the max, it can be ignored.
   *
   * @param max  the largest scan number that can be used by the lane
   */
  public abstract void setMaxScan(int max);

  /**
   * Writes all of the information this class needs to store in order
   * to be recreated. This will be used for things like storing the
   * class in a file. Therefore, the write should output enough information
   * so that <code>read</code> can recreate the essential properties of this
   * class.
   *
   * @param out  the destination to write the data to.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public abstract void write(DataOutputStream out) throws IOException;

  /**
   * Reads in the properties of this class from the specified input stream.
   * The stream data should have been created by <code>write</code>. This
   * will retrieve this classes state from the input stream. All of the
   * current data in this class will be replaced by the data from the 
   * stream.
   *
   * @param in  the input stream with the data for the class.
   *
   * @exception IOException  occurs when a problem is encountered when
   *     writing to the stream.
   */
  public abstract void read(DataInputStream in) 
       throws IOException;

  /**
   * Gives an object where every field has been copied from this object
   * to the new object.
   *
   * @return a copy of this object.
   */
  public Object clone()
  {
    try
      {
	return super.clone();
      }
    catch(CloneNotSupportedException e)
      {
	System.err.println("Could not clone Sizing Function. " + 
			   e.getMessage());
      }
    return null;
  }
}

