#!/bin/csh

#Version  8/27/2007
# Run discrete data parsimony programs as a command
#Synopsis: discpars.csh infile utree ufn method bseed replicates percent\
#             TCONMETH PARSTYPE THRESHOLD outgroup jumble jseed numjum\
#             termout printdata outfile treefile

#Convert arguments to variables
set INFILE        = $1
set UTREE         = $2
set UFN           = $3
set METHOD        = $4
set BSEED         = $5
set REPLICATES    = $6
set PERCENT       = $7
set TCONMETH      = $8
set PARSTYPE      = $9
set THRESHOLD     = $10
set OUTGROUP      = $11
set JUMBLE        = $12
set JSEED         = $13
set NUMJUM        = $14
set TERMOUT       = $15
set PRINTDATA     = $16
set OUTFILE       = $17
set TREEFILE      = $18


# RM_CMD - command to be used for removing files and directories
if (-e /usr/bin/rm) then
   set RM_CMD = /usr/bin/rm
else
   if (-e /bin/rm) then
      set RM_CMD = /bin/rm
   else
      set RM_CMD = rm
   endif
endif


# Make a temporary directory in which to run the program
set TEMPDIR = `birchtmpdir.sh`
echo 'TEMPDIR ' $TEMPDIR
if ( -z "$TEMPDIR" ) then
   set TEMPDIR = DISCPARS.$$
else
   set TEMPDIR = $TEMPDIR/DISCPARS.$$
endif
echo 'TEMPDIR ' $TEMPDIR
mkdir $TEMPDIR

# Remember where we started
set STARTDIR = $PWD
echo 'Starting Directory: ' $STARTDIR

cp $INFILE $TEMPDIR/infile.temp
if ($UTREE == 'y') then
   # Turn off bootstrapping when evaluating a user tree.
   # If bootstrapping is on, an empty file may be generated
   # which could cause drawtree or drawgram to loop infinitely
   set METHOD = n
   
   # Make sure that treefile begins with number of trees on first
   # line of file. If first line in file has parentheses, the
   # number must be added.
   if (`head -1 $UFN | grep '('` != "" ) then
      grep ';' $UFN | wc -l >> $TEMPDIR/intree
   endif
   cat $UFN >> $TEMPDIR/intree
endif

cd $TEMPDIR

# Vestigial code leftover from Phylip 3.5. Not even sure why it was needed.
#set FIRSTLINE = `head -1 infile.temp`
#set NUMSEQ = $FIRSTLINE[1]
#mv infile.temp infile.temp.temp
#echo $FIRSTLINE > infile.temp
#tail +2 infile.temp.temp >> infile.temp
#$RM_CMD infile.temp.temp

if ($UTREE == 'y') then
   echo 'u' >> TREEPARAMFILE
endif

echo "---------------------  DISCRETE DATA PARSIMONY METHODS ---------------------" >> MSGFILE
echo "" >> MSGFILE

#----------------- generated resampled datasets, if specified  -----
# Choose resampling method

# Random seeds, odd, of the form 4n + 1
@ tempbseed = ( ( ( $BSEED / 4 ) * 4 ) + 1 )

switch ($METHOD)
  case "n":
    echo " " >> MSGFILE
    cp infile.temp infile
    breaksw
  case "b":
    echo RESAMPLING\: Bootstrap,  $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    if ( $PERCENT < 100 ) echo 'Partial Resampling: ' $PERCENT  'percent of sites sampled' >> MSGFILE
    cp infile.temp infile
    seqboot.csh infile.temp m $tempbseed b $REPLICATES $PERCENT 1 yes 0 weights
    echo m >> TREEPARAMFILE
    echo w >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "d" :
    echo RESAMPLING\: Delete-half Jacknifing, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    if ( $PERCENT < 100 ) echo 'Partial Resampling: ' $PERCENT  'percent of sites sampled' >> MSGFILE
    cp infile.temp infile
    seqboot.csh infile.temp m $tempbseed d $REPLICATES $PERCENT 1 yes 0 weights
    echo m >> TREEPARAMFILE
    echo w >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "ps":
    echo RESAMPLING\: Permute species for each character, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    seqboot.csh infile.temp m $tempbseed ps $REPLICATES $PERCENT 1 no 0 infile
    echo m >> TREEPARAMFILE
    echo d >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "po":
    echo RESAMPLING\: Permute character order, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    seqboot.csh infile.temp m $tempbseed po $REPLICATES $PERCENT 1 no 0 infile
    echo m >> TREEPARAMFILE
    echo d >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw
  case "pw":
    echo RESAMPLING\: Permute within species, $REPLICATES REPLICATES, SEED\=$BSEED >>MSGFILE
    seqboot.csh infile.temp m $tempbseed pw $REPLICATES $PERCENT 1 no 0 infile
    echo m >> TREEPARAMFILE
    echo d >> TREEPARAMFILE
    echo $REPLICATES >> TREEPARAMFILE
    breaksw        
endsw


#----------------- generate keyboard input to send to parsimony program -----

# Choose program and set program-specific parameters
switch ($TCONMETH)
  case "d"
    set PROGRAM = dollop
    breaksw
  case "p"
    set PROGRAM = dollop
    echo p >> TREEPARAMFILE
    breaksw
  case "c"
    set PROGRAM = mix
    echo p >> TREEPARAMFILE
    breaksw
  case "w"
    set PROGRAM = mix
    breaksw
# x requires a file called 'mixed', which specifies, for each character,
# which method to use. Tricky to implement, at present, so we'll leave
# this commented out.    
#  case "x"
#    set PROGRAM = mix
#    echo x >> TREEPARAMFILE
#    breaksw    
  default:
    set PROGRAM = dollop
    breaksw
endsw


# Jumble - When multiple datasets are analyzed, DNAPARS automatically
# jumbles, and prompts for a random number seed for jumbling. Othersise,
# jumbling must be explicitly set.

# Random seed, odd, of the form 4n + 1
@ tempjseed = ( ( ( $JSEED / 4 ) * 4 ) + 1 )

if ($METHOD == "n") then
   if ($JUMBLE == J) then 
      echo JUMBLING SEQUENCE ORDER $NUMJUM ITERATIONS, SEED\=$tempjseed >> MSGFILE
      echo  j >> TREEPARAMFILE
      echo $tempjseed  >> TREEPARAMFILE
      echo $NUMJUM >> TREEPARAMFILE
   endif
else
   echo JUMBLING SEQUENCE ORDER $NUMJUM ITERATIONS, SEED\=$tempjseed >> MSGFILE
   echo $tempjseed  >> TREEPARAMFILE
   echo $NUMJUM >> TREEPARAMFILE
endif


# Outgroup
if ($PROGRAM == mix) then
  set FIRSTLINE = `head -1 infile`
  set NUMSEQ = $FIRSTLINE[1]
  if (($OUTGROUP > 1) && ($OUTGROUP <= $NUMSEQ)) then
     echo o >> TREEPARAMFILE
     echo $OUTGROUP >> TREEPARAMFILE 
  else
     set OUTGROUP = 1
  endif
endif

# Threshold or ordinary parsimony
if ($PARSTYPE ==  t) then
   echo 'Using threshold parsimony, THRESHOLD= ' $THRESHOLD >> MSGFILE
   if ($THRESHOLD < 2 ) set THRESHOLD = 2
   echo $PARSTYPE >> TREEPARAMFILE
   echo $THRESHOLD >> TREEPARAMFILE
endif

# Should sequence data be printed?
if ($PRINTDATA ==  y) echo 1 >> TREEPARAMFILE

# When resampling, turn off printing trees to outfile

if (($METHOD == 'b') || ($METHOD == 'd') || ($METHOD == 'ps') || ($METHOD == 'po') || ($METHOD == 'pw')) then
   echo 3 >> TREEPARAMFILE
endif

#accept current settings and do the analysis
echo y  >> TREEPARAMFILE


#----------------- Run DOLLOP OR MIX  -----

echo `time nice +8 $PROGRAM < TREEPARAMFILE > $TERMOUT` > TIMEFILE
echo Execution times on `hostname`\: `cat TIMEFILE` >> outfile


#----------- Return results to calling directory----------------
# When using resampling, filter the treefile through
# consense to generate an unrooted consensus tree.
if (($METHOD == 'b') || ($METHOD == 'd') || ($METHOD == 'ps') || ($METHOD == 'po') || ($METHOD == 'pw')) then
	echo " " >> outfile
	echo '-------------------------------------------' >> outfile
	if ($PROGRAM == "dollop") then
	   set ROOTEDTREE = "y"
	else
	   set ROOTEDTREE = "n"
	endif

      consense.csh outtree e 1 $OUTGROUP $ROOTEDTREE outfile.consense constree
      cat MSGFILE outfile outfile.consense  > $STARTDIR/$OUTFILE
      mv outtree $STARTDIR/$TREEFILE.alltrees	# save the raw trees, not the consensus tree
      mv constree $STARTDIR/$TREEFILE
      echo '' >> $STARTDIR/$OUTFILE
      echo '' >> $STARTDIR/$OUTFILE
      echo '>>>> THIS TREEFILE IS A CONSENSUS TREE' >> $STARTDIR/$OUTFILE
      echo '>>>> TO GENERATE BRANCH LENGTHS' >> $STARTDIR/$OUTFILE
      echo '>>>> USE TREE FILE AS INPUT FOR DNAML OR OTHER PROGRAM' >> $STARTDIR/$OUTFILE
      echo '>>>> USING THE USERTREE OPTION' >> $STARTDIR/$OUTFILE
else
        mv outtree $STARTDIR/$TREEFILE
	cat MSGFILE outfile > $STARTDIR/$OUTFILE
endif

cd $STARTDIR
$RM_CMD -rf $TEMPDIR

echo $PROGRAM completed.





